#!/usr/bin/env python3
"""
Quodd Stock Quote Fetcher

Fetches real-time stock quotes via the Quodd API with token caching.
"""

import argparse
import json
import os
import sys
from datetime import datetime, timedelta, timezone
from pathlib import Path
from urllib.request import Request, urlopen
from urllib.error import URLError, HTTPError


CACHE_DIR = Path.home() / ".openclaw" / "credentials"
TOKEN_CACHE_FILE = CACHE_DIR / "quodd-token.json"
TOKEN_TTL_HOURS = 20

TOKEN_URL = "https://vor.quodd.com/vor/quodd/login/quodddst/token"
QUOTE_URL_TEMPLATE = "https://custsnap.quodd.com/quoddsnap/c/quodddst/t/{token}"

TIMEOUT_SECONDS = 30


def get_credentials():
    """Get Quodd credentials from environment variables."""
    username = os.environ.get("QUODD_USERNAME")
    password = os.environ.get("QUODD_PASSWORD")

    if not username or not password:
        print("Error: Missing Quodd credentials.", file=sys.stderr)
        print("", file=sys.stderr)
        print("Set environment variables:", file=sys.stderr)
        print("  export QUODD_USERNAME='your_username'", file=sys.stderr)
        print("  export QUODD_PASSWORD='your_password'", file=sys.stderr)
        sys.exit(1)

    return username, password


def load_cached_token():
    """Load token from cache if valid."""
    if not TOKEN_CACHE_FILE.exists():
        return None

    try:
        with open(TOKEN_CACHE_FILE, "r") as f:
            data = json.load(f)

        expires_at = datetime.fromisoformat(data["expires_at"])
        now = datetime.now(timezone.utc)

        # Check if token is expired or will expire within 1 hour
        if expires_at > now + timedelta(hours=1):
            return data["token"]

        return None
    except (json.JSONDecodeError, KeyError, ValueError):
        return None


def save_token_to_cache(token):
    """Save token to cache with expiration timestamp."""
    CACHE_DIR.mkdir(parents=True, exist_ok=True)

    expires_at = datetime.now(timezone.utc) + timedelta(hours=TOKEN_TTL_HOURS)

    data = {
        "token": token,
        "expires_at": expires_at.isoformat()
    }

    with open(TOKEN_CACHE_FILE, "w") as f:
        json.dump(data, f, indent=2)


def fetch_token(username, password):
    """Fetch a new token from the Quodd API."""
    headers = {
        "username": username,
        "password": password,
        "Content-Type": "application/json"
    }

    req = Request(TOKEN_URL, data=b"", headers=headers, method="POST")

    try:
        with urlopen(req, timeout=TIMEOUT_SECONDS) as response:
            data = json.loads(response.read().decode("utf-8"))
            return data.get("token")
    except HTTPError as e:
        if e.code == 401:
            print("Error: Authentication failed.", file=sys.stderr)
            print("Please check your QUODD_USERNAME and QUODD_PASSWORD.", file=sys.stderr)
        else:
            print(f"Error: HTTP {e.code} when fetching token.", file=sys.stderr)
        sys.exit(1)
    except URLError as e:
        print(f"Error: Network error when fetching token: {e.reason}", file=sys.stderr)
        sys.exit(1)
    except TimeoutError:
        print("Error: Request timed out when fetching token.", file=sys.stderr)
        sys.exit(1)


def get_token(use_cache=True):
    """Get a valid token, using cache if available."""
    if use_cache:
        cached_token = load_cached_token()
        if cached_token:
            return cached_token

    username, password = get_credentials()
    token = fetch_token(username, password)

    if not token:
        print("Error: Failed to obtain token from Quodd API.", file=sys.stderr)
        sys.exit(1)

    save_token_to_cache(token)
    return token


def is_empty_value(value):
    """Check if a value represents missing/empty data."""
    if value is None:
        return True
    if isinstance(value, str):
        stripped = value.strip()
        # Handle empty strings and placeholder values like "----", "--", "N/A"
        if not stripped or stripped.replace("-", "") == "" or stripped.upper() in ("N/A", "NA", "NULL"):
            return True
    if isinstance(value, (int, float)) and value == 0:
        # Zero often indicates no data for prices/timestamps
        return True
    return False


def convert_4d_price(price_4d):
    """Convert 4D format price to decimal (e.g., 2561200 -> 256.12)."""
    if is_empty_value(price_4d):
        return None
    try:
        price = float(price_4d) / 10000.0
        # Sanity check - prices should be positive and reasonable
        if price <= 0:
            return None
        return price
    except (TypeError, ValueError):
        return None


def format_price(price):
    """Format price for display."""
    if price is None:
        return "-"
    return f"{price:.2f}"


def format_time(time_str):
    """Format time string for display."""
    if not time_str:
        return "-"
    return time_str


def fetch_quotes(token, tickers):
    """Fetch quotes for the given tickers."""
    url = QUOTE_URL_TEMPLATE.format(token=token)

    # Append .d suffix to tickers
    ticker_list = ",".join(f"{t.upper()}.d" for t in tickers)

    payload = json.dumps({"tickers": ticker_list}).encode("utf-8")

    headers = {
        "Content-Type": "application/json"
    }

    req = Request(url, data=payload, headers=headers, method="POST")

    try:
        with urlopen(req, timeout=TIMEOUT_SECONDS) as response:
            return json.loads(response.read().decode("utf-8"))
    except HTTPError as e:
        if e.code == 401:
            print("Error: Token expired or invalid. Try with --no-cache.", file=sys.stderr)
        else:
            print(f"Error: HTTP {e.code} when fetching quotes.", file=sys.stderr)
        sys.exit(1)
    except URLError as e:
        print(f"Error: Network error when fetching quotes: {e.reason}", file=sys.stderr)
        sys.exit(1)
    except TimeoutError:
        print("Error: Request timed out when fetching quotes.", file=sys.stderr)
        sys.exit(1)


def parse_timestamp(ts_value):
    """Parse a timestamp (Unix ms/s or ISO string) to local datetime."""
    if is_empty_value(ts_value):
        return None
    try:
        if isinstance(ts_value, (int, float)):
            # Unix timestamp (seconds or milliseconds)
            ts = ts_value if ts_value < 10000000000 else ts_value / 1000
            # Sanity check - timestamp should be reasonable (after year 2000)
            if ts < 946684800:  # Jan 1, 2000
                return None
            return datetime.fromtimestamp(ts)  # Local time
        else:
            # Handle string timestamps
            ts_str = str(ts_value).strip()
            if not ts_str or ts_str.replace("-", "") == "":
                return None
            dt = datetime.fromisoformat(ts_str.replace("Z", "+00:00"))
            return dt.astimezone()  # Convert to local time
    except (ValueError, OSError, TypeError):
        return None


def parse_quote(quote_data):
    """Parse a quote response into a structured format."""
    ticker = quote_data.get("ticker", "")
    # Remove .d suffix if present (case-insensitive)
    if ticker.lower().endswith(".d"):
        ticker = ticker[:-2]

    # Parse quote time
    quote_dt = parse_timestamp(quote_data.get("quote_time"))
    date_str = quote_dt.strftime("%m/%d/%y") if quote_dt else ""
    time_str = quote_dt.strftime("%H:%M") if quote_dt else ""

    # Parse after hours time
    ah_dt = parse_timestamp(quote_data.get("ext_trade_time"))
    ah_time_str = ah_dt.strftime("%H:%M:%S") if ah_dt else ""

    return {
        "symbol": ticker,
        "date": date_str,
        "time": time_str,
        "high": convert_4d_price(quote_data.get("day_high_4d")),
        "low": convert_4d_price(quote_data.get("day_low_4d")),
        "close": convert_4d_price(quote_data.get("last_price_4d")),
        "after_hours_time": ah_time_str,
        "after_hours_price": convert_4d_price(quote_data.get("ext_last_price_4d"))
    }


def output_text(quotes):
    """Output quotes in text table format."""
    print("Quodd Stock Quotes")
    print(f"{'Symbol':<8} {'Date':<11} {'Time':<9} {'High':>9} {'Low':>9} {'Close':>9} {'AH Time':<11} {'AH Price':>9}")
    print("-" * 79)

    for q in quotes:
        print(
            f"{q['symbol']:<8} "
            f"{q['date']:<11} "
            f"{q['time']:<9} "
            f"{format_price(q['high']):>9} "
            f"{format_price(q['low']):>9} "
            f"{format_price(q['close']):>9} "
            f"{format_time(q['after_hours_time']):<11} "
            f"{format_price(q['after_hours_price']):>9}"
        )


def output_json(quotes):
    """Output quotes in JSON format."""
    # Convert to ISO date format for JSON
    json_quotes = []
    for q in quotes:
        json_quote = {
            "symbol": q["symbol"],
            "date": q["date"],
            "time": q["time"],
            "high": q["high"],
            "low": q["low"],
            "close": q["close"],
            "after_hours_time": q["after_hours_time"] if q["after_hours_time"] else None,
            "after_hours_price": q["after_hours_price"]
        }
        json_quotes.append(json_quote)

    print(json.dumps({"quotes": json_quotes}, indent=2))


def main():
    parser = argparse.ArgumentParser(
        description="Fetch real-time stock quotes via Quodd API"
    )
    parser.add_argument(
        "tickers",
        nargs="+",
        help="Stock ticker symbols (e.g., AAPL MSFT META)"
    )
    parser.add_argument(
        "--format",
        choices=["text", "json"],
        default="text",
        help="Output format (default: text)"
    )
    parser.add_argument(
        "--no-cache",
        action="store_true",
        help="Force token refresh, ignoring cache"
    )

    args = parser.parse_args()

    # Get token
    token = get_token(use_cache=not args.no_cache)

    # Fetch quotes
    response = fetch_quotes(token, args.tickers)

    # Handle response - could be a list or dict with data
    if isinstance(response, list):
        quote_list = response
    elif isinstance(response, dict):
        quote_list = response.get("data", response.get("quotes", [response]))
    else:
        quote_list = []

    if not quote_list:
        print("No quotes returned.", file=sys.stderr)
        sys.exit(1)

    # Parse quotes
    quotes = []
    for item in quote_list:
        if isinstance(item, dict):
            quotes.append(parse_quote(item))

    if not quotes:
        print("No valid quotes found.", file=sys.stderr)
        sys.exit(1)

    # Output
    if args.format == "json":
        output_json(quotes)
    else:
        output_text(quotes)


if __name__ == "__main__":
    main()
