#!/usr/bin/env node

// Querit Search CLI — queries the Querit.ai search API
// No external dependencies; uses Node.js built-in fetch (18+)

const API_URL = "https://api.querit.ai/v1/search";
const MAX_QUERY_LENGTH = 72;
const DEFAULT_COUNT = 5;

function usage() {
  console.error(`Usage: search.js <query> [options]

Options:
  -n <count>                Number of results (default: ${DEFAULT_COUNT}, max: 100)
  --lang <language>         Language filter (english, japanese, korean, german, french, spanish, portuguese)
  --country <country>       Country filter (e.g. "united states", japan, germany)
  --date <range>            Date filter (d1=past day, w1=past week, m1=past month, y1=past year)
  --site-include <domain>   Only include results from this domain (repeatable)
  --site-exclude <domain>   Exclude results from this domain (repeatable)
  --content                 Also fetch and extract page content as markdown
  --json                    Output raw JSON

Environment:
  QUERIT_API_KEY            Required. Your Querit API key.`);
}

function parseArgs(argv) {
  const args = {
    query: null,
    count: DEFAULT_COUNT,
    lang: null,
    country: null,
    date: null,
    siteInclude: [],
    siteExclude: [],
    content: false,
    json: false,
  };

  let i = 0;
  while (i < argv.length) {
    const arg = argv[i];

    if (arg === "-n" || arg === "--count") {
      args.count = parseInt(argv[++i], 10);
      if (isNaN(args.count) || args.count < 1) {
        console.error("Error: -n must be a positive integer");
        process.exit(1);
      }
      if (args.count > 100) {
        console.error("Error: -n cannot exceed 100");
        process.exit(1);
      }
    } else if (arg === "--lang") {
      args.lang = argv[++i];
    } else if (arg === "--country") {
      args.country = argv[++i];
    } else if (arg === "--date") {
      args.date = argv[++i];
    } else if (arg === "--site-include") {
      args.siteInclude.push(argv[++i]);
    } else if (arg === "--site-exclude") {
      args.siteExclude.push(argv[++i]);
    } else if (arg === "--content") {
      args.content = true;
    } else if (arg === "--json") {
      args.json = true;
    } else if (arg === "--help" || arg === "-h") {
      usage();
      process.exit(0);
    } else if (!arg.startsWith("-")) {
      args.query = arg;
    } else {
      console.error(`Unknown option: ${arg}`);
      usage();
      process.exit(1);
    }
    i++;
  }

  return args;
}

function buildPayload(args) {
  let query = args.query;

  if (query.length > MAX_QUERY_LENGTH) {
    console.error(
      `Warning: Query truncated from ${query.length} to ${MAX_QUERY_LENGTH} characters`
    );
    query = query.slice(0, MAX_QUERY_LENGTH);
  }

  const payload = {
    query,
    count: args.count,
  };

  const filters = {};

  if (args.lang) {
    filters.languages = { include: [args.lang.toLowerCase()] };
  }

  if (args.country) {
    filters.geo = { countries: { include: [args.country.toLowerCase()] } };
  }

  if (args.date) {
    filters.timeRange = { date: args.date };
  }

  if (args.siteInclude.length > 0) {
    filters.sites = filters.sites || {};
    filters.sites.include = args.siteInclude;
  }

  if (args.siteExclude.length > 0) {
    filters.sites = filters.sites || {};
    filters.sites.exclude = args.siteExclude;
  }

  if (Object.keys(filters).length > 0) {
    payload.filters = filters;
  }

  return payload;
}

function extractDomain(url) {
  try {
    return new URL(url).hostname;
  } catch {
    return null;
  }
}

function formatResults(items) {
  const lines = [];

  for (let i = 0; i < items.length; i++) {
    const item = items[i];
    const num = i + 1;
    const domain = extractDomain(item.url);

    lines.push(`${num}. ${item.title || "(no title)"}`);
    lines.push(`   ${item.url}`);
    if (domain) {
      lines.push(`   Site: ${domain}`);
    }
    if (item.page_age) {
      lines.push(`   Age: ${item.page_age}`);
    }
    if (item.snippet) {
      lines.push(`   ${item.snippet}`);
    }
    lines.push("");
  }

  return lines.join("\n");
}

async function fetchContent(url) {
  const { execFile } = await import("node:child_process");
  const { promisify } = await import("node:util");
  const execFileAsync = promisify(execFile);
  const scriptDir = new URL(".", import.meta.url).pathname;

  try {
    const { stdout } = await execFileAsync(
      process.execPath,
      [`${scriptDir}content.js`, url],
      { timeout: 30_000 }
    );
    return stdout.trim();
  } catch (err) {
    return `(Failed to extract content: ${err.message})`;
  }
}

async function main() {
  const argv = process.argv.slice(2);

  if (argv.length === 0) {
    usage();
    process.exit(1);
  }

  const args = parseArgs(argv);

  if (!args.query) {
    console.error("Error: No search query provided");
    usage();
    process.exit(1);
  }

  const apiKey = process.env.QUERIT_API_KEY;
  if (!apiKey) {
    console.error(
      "Error: QUERIT_API_KEY environment variable is not set.\n" +
        "Get a free API key at https://querit.ai and set it:\n" +
        '  export QUERIT_API_KEY="your-key-here"'
    );
    process.exit(1);
  }

  const payload = buildPayload(args);

  let data;
  try {
    const res = await fetch(API_URL, {
      method: "POST",
      headers: {
        Authorization: `Bearer ${apiKey}`,
        "Content-Type": "application/json",
      },
      body: JSON.stringify(payload),
    });

    data = await res.json();
  } catch (err) {
    console.error(`Error: API request failed — ${err.message}`);
    process.exit(1);
  }

  if (data.error_code && data.error_code !== 200) {
    console.error(
      `Error: API returned error ${data.error_code}: ${data.error_msg || "Unknown error"}`
    );
    process.exit(1);
  }

  const items = data.results?.result || [];

  if (items.length === 0) {
    console.log("No results found.");
    process.exit(0);
  }

  if (args.json) {
    console.log(JSON.stringify(items, null, 2));
    process.exit(0);
  }

  console.log(formatResults(items));

  if (args.content) {
    console.log("--- Fetching page content ---\n");
    for (let i = 0; i < items.length; i++) {
      const item = items[i];
      console.log(`### ${i + 1}. ${item.title || "(no title)"}`);
      console.log(`URL: ${item.url}\n`);
      const md = await fetchContent(item.url);
      console.log(md);
      console.log("\n---\n");
    }
  }
}

main();
