#!/usr/bin/env bash
set -euo pipefail

# Querit Search — OpenClaw Skill Installer
# Usage: curl -fsSL https://raw.githubusercontent.com/interskh/querit-search/main/install.sh | bash

SKILL_DIR="${HOME}/.openclaw/skills/querit-search"
REPO_BASE="https://raw.githubusercontent.com/interskh/querit-search/main"
FILES="SKILL.md search.js content.js package.json _meta.json"

info() { printf '\033[1;34m%s\033[0m\n' "$1"; }
success() { printf '\033[1;32m%s\033[0m\n' "$1"; }
error() { printf '\033[1;31mError: %s\033[0m\n' "$1" >&2; exit 1; }

# Check prerequisites
command -v node >/dev/null 2>&1 || error "Node.js is required but not installed"
command -v npm >/dev/null 2>&1 || error "npm is required but not installed"

NODE_VERSION=$(node -v | sed 's/v//' | cut -d. -f1)
if [ "$NODE_VERSION" -lt 18 ]; then
  error "Node.js 18+ required (found v${NODE_VERSION})"
fi

info "Installing Querit Search skill to ${SKILL_DIR}..."

# Create skill directory
mkdir -p "$SKILL_DIR"

# Download skill files
for file in $FILES; do
  info "  Downloading ${file}..."
  curl -fsSL "${REPO_BASE}/${file}" -o "${SKILL_DIR}/${file}" || error "Failed to download ${file}"
done

# Make scripts executable
chmod +x "${SKILL_DIR}/search.js" "${SKILL_DIR}/content.js"

# Install npm dependencies
info "  Installing npm dependencies..."
cd "$SKILL_DIR"
npm ci --production --silent 2>/dev/null || npm install --production --silent || error "npm install failed"

# Verify installation
if [ ! -d "${SKILL_DIR}/node_modules" ]; then
  error "node_modules not created — npm install may have failed"
fi

echo ""
success "Querit Search skill installed successfully!"
echo ""
echo "Next steps:"
echo "  1. Get a free API key at https://querit.ai"
echo "  2. Set it in your environment:"
echo "       export QUERIT_API_KEY=\"your-key-here\""
echo ""
echo "  Or add to your OpenClaw config (~/.openclaw/openclaw.json):"
echo '       "skills": { "entries": { "querit-search": { "apiKey": "your-key-here" } } }'
echo ""
echo "  3. Start OpenClaw and ask it to search the web!"
