"""
Output formatter — turns compiled feature dicts into structured text
for the agent to read (printed to stdout).

Strategy-aware: format_full() produces a 3-tier hierarchical output
(primary TF prominent, contextual TFs condensed, all TFs compact)
with per-strategy noise gating, CVD filtering, and risk thresholds.
"""

from typing import Any, Dict

from .analysis.strategies import CVD_DISPLAY, CVD_SHORT_LABEL

ALL_TIMEFRAMES = ["1m", "5m", "15m", "30m", "1h", "4h", "D", "W"]


# ── Formatting helpers ─────────────────────────────────────────────


def _fmt(val: Any, decimals: int = 2) -> str:
    """Format a numeric value, returning '—' for None/missing."""
    if val is None:
        return "—"
    if isinstance(val, float):
        return f"{val:.{decimals}f}"
    return str(val)


def _pct(val: Any) -> str:
    if val is None:
        return "—"
    return f"{val:+.2f}%"


def _usd(val: Any) -> str:
    if val is None:
        return "—"
    v = float(val)
    if abs(v) >= 1_000_000:
        return f"${v / 1_000_000:.2f}M"
    if abs(v) >= 1_000:
        return f"${v / 1_000:.1f}K"
    return f"${v:.2f}"


def _gate(ok: Any) -> str:
    if ok is None:
        return "N/A"
    return "PASS" if ok else "FAIL"


def _macd_signal(tf_data: dict, tf: str) -> str:
    """Derive MACD direction from histogram value."""
    h = tf_data.get(f"macd.histogram_{tf}")
    if h is None:
        return "—"
    return "bull" if h > 0 else "bear"


def _ema_cross(tf_data: dict, tf: str) -> str:
    """Derive EMA 20/50 cross direction."""
    ema20 = tf_data.get(f"ema20_{tf}")
    ema50 = tf_data.get(f"ema50_{tf}")
    if ema20 is None or ema50 is None:
        return "—"
    return "bullish" if ema20 > ema50 else "bearish"


# ── Indicator formatting tiers ─────────────────────────────────────


def _format_tf_full(lines: list, tf_data: dict, tf: str) -> None:
    """Tier 1: Full indicator detail for primary timeframe."""
    if "error" in tf_data:
        lines.append(f"  {tf_data['error']}")
        return
    if not tf_data:
        lines.append("  No data")
        return

    rsi = _fmt(tf_data.get(f"rsi.level_{tf}"))
    macd_h = _fmt(tf_data.get(f"macd.histogram_{tf}"), 4)
    macd_sig = _macd_signal(tf_data, tf)
    bb_pct = _fmt(tf_data.get(f"boll.pct_{tf}"))
    atr_pct = _fmt(tf_data.get(f"atr.pct_{tf}"))
    adx = _fmt(tf_data.get(f"adx.level_{tf}"))
    chop = _fmt(tf_data.get(f"chop.index_{tf}"))
    cci = _fmt(tf_data.get(f"cci.level_{tf}"))
    willr = _fmt(tf_data.get(f"willr.level_{tf}"))

    lines.append(
        f"  RSI: {rsi}  MACD: {macd_h} ({macd_sig})  "
        f"BB%: {bb_pct}  ATR%: {atr_pct}"
    )

    di_str = ""
    plus_di = tf_data.get(f"adx.plus_di_{tf}")
    minus_di = tf_data.get(f"adx.minus_di_{tf}")
    if plus_di is not None and minus_di is not None:
        di_str = f" (+DI:{plus_di:.0f} -DI:{minus_di:.0f})"
    lines.append(f"  ADX: {adx}{di_str}  Chop: {chop}  CCI: {cci}  W%R: {willr}")

    ema20 = tf_data.get(f"ema20_{tf}")
    ema50 = tf_data.get(f"ema50_{tf}")
    if ema20 is not None and ema50 is not None:
        cross = _ema_cross(tf_data, tf)
        lines.append(f"  EMA20: {ema20:,.0f}  EMA50: {ema50:,.0f} ({cross} cross)")

    stoch_k = _fmt(tf_data.get(f"stoch.k_{tf}"))
    stoch_d = _fmt(tf_data.get(f"stoch.d_{tf}"))
    vol_ratio = _fmt(tf_data.get(f"vol.ratio_{tf}"))
    lines.append(f"  Stoch: K={stoch_k} D={stoch_d}  Vol Ratio: {vol_ratio}")


def _format_tf_condensed(lines: list, tf_data: dict, tf: str) -> None:
    """Tier 2: Condensed one-line summary for contextual timeframes."""
    if "error" in tf_data:
        lines.append(f"  {tf_data['error']}")
        return
    if not tf_data:
        lines.append("  No data")
        return

    rsi = _fmt(tf_data.get(f"rsi.level_{tf}"))
    macd_h = _fmt(tf_data.get(f"macd.histogram_{tf}"), 4)
    macd_sig = _macd_signal(tf_data, tf)
    adx = _fmt(tf_data.get(f"adx.level_{tf}"))
    atr_pct = _fmt(tf_data.get(f"atr.pct_{tf}"))

    lines.append(
        f"  RSI: {rsi}  MACD: {macd_h} ({macd_sig})  "
        f"ADX: {adx}  ATR%: {atr_pct}"
    )


def _format_tf_oneline(lines: list, tf_data: dict, tf: str) -> None:
    """Tier 3: Compact one-liner for multi-TF summary."""
    if "error" in tf_data or not tf_data:
        return

    rsi_val = tf_data.get(f"rsi.level_{tf}")
    rsi = f"{rsi_val:.0f}" if rsi_val is not None else "—"
    macd_sig = _macd_signal(tf_data, tf)
    adx_val = tf_data.get(f"adx.level_{tf}")
    adx = f"{adx_val:.0f}" if adx_val is not None else "—"
    chop_val = tf_data.get(f"chop.index_{tf}")
    chop = f"{chop_val:.0f}" if chop_val is not None else "—"

    lines.append(f"  [{tf:>3s}]  RSI:{rsi:>3s}  MACD:{macd_sig:>4s}  ADX:{adx:>3s}  Chop:{chop:>3s}")


# ── Full Analysis (strategy-aware, 3-tier hierarchy) ───────────────


def format_full(data: Dict[str, Any]) -> str:
    """Format a full analysis dict with strategy-aware 3-tier hierarchy."""
    lines: list[str] = []

    t = data.get("ticker", {})
    ind = data.get("indicators", {})
    strat = data.get("strategy", {})

    strategy_name = strat.get("name", "swing").upper()
    primary_tf = strat.get("primary_tf", "15m")
    lower_tf = strat.get("lower_tf", "5m")
    higher_tf = strat.get("higher_tf", "4h")
    full_micro = strat.get("full_microstructure", False)
    fund_weight = strat.get("fund_weight", [])
    cvd_keys = strat.get("cvd_keys", [])

    # ── Section 1: Strategy Context ──────────────────────────────────
    lines.append(f"=== {t.get('symbol', '???')} {strategy_name} Analysis ===")
    lines.append(
        f"Strategy: {strategy_name} "
        f"({primary_tf} chart, {strat.get('hold_period', '?')} holds)"
    )
    if fund_weight:
        lines.append(f"Fundamentals weight: {' > '.join(fund_weight)}")
    lines.append(
        f"Price: ${t.get('last_price', 0):,.2f}  "
        f"(Mark: ${t.get('mark_price', 0):,.2f})"
    )
    lines.append(f"24h Volume: {_usd(t.get('turnover_24h'))}")
    lines.append(
        f"Funding: {_fmt(t.get('funding_rate'), 6)}  |  "
        f"OI: {_usd(t.get('open_interest'))}"
    )
    lines.append("")

    # ── Section 2: Primary Timeframe (prominent) ─────────────────────
    lines.append(f"--- Primary: {primary_tf} ---")
    _format_tf_full(lines, ind.get(primary_tf, {}), primary_tf)
    lines.append("")

    # ── Section 3: Contextual Timeframes (condensed) ─────────────────
    lines.append(f"--- Context: {lower_tf} (lower) ---")
    _format_tf_condensed(lines, ind.get(lower_tf, {}), lower_tf)
    lines.append(f"--- Context: {higher_tf} (higher) ---")
    _format_tf_condensed(lines, ind.get(higher_tf, {}), higher_tf)
    lines.append("")

    # ── Section 4: Multi-TF Summary (compact) ────────────────────────
    lines.append("--- All Timeframes ---")
    for tf in ALL_TIMEFRAMES:
        _format_tf_oneline(lines, ind.get(tf, {}), tf)
    lines.append("")

    # ── Section 5: Orderbook ─────────────────────────────────────────
    lines.append("--- Orderbook ---")
    ms = data.get("microstructure", {})
    if full_micro:
        # FAST: full orderbook detail
        lines.append(f"  Microprice: ${_fmt(ms.get('ob.microprice'))}")
        lines.append(f"  Spread: {_fmt(ms.get('ob.spread_bps'))} bps")
        lines.append(
            f"  QI L1: {_fmt(ms.get('ob.qi_l1'))}  "
            f"QI L5: {_fmt(ms.get('ob.qi_l5'))}"
        )
        lines.append(
            f"  Depth L1: {_usd(ms.get('ob.depth_usd_l1'))}  "
            f"L5: {_usd(ms.get('ob.depth_usd_l5'))}"
        )
        lines.append(
            f"  Slope Bid L10: {_fmt(ms.get('ob.slope_bid_l10'), 4)}  "
            f"Ask: {_fmt(ms.get('ob.slope_ask_l10'), 4)}"
        )
    else:
        # Non-FAST: condensed (3 fields)
        lines.append(f"  Spread: {_fmt(ms.get('ob.spread_bps'))} bps")
        lines.append(f"  QI L1: {_fmt(ms.get('ob.qi_l1'))}")
        lines.append(f"  Depth L5: {_usd(ms.get('ob.depth_usd_l5'))}")
    lines.append("")

    # ── Section 6: Trade Flow (strategy-filtered CVD) ────────────────
    cvd_label = "CVD"
    if cvd_keys:
        cvd_label = CVD_SHORT_LABEL.get(cvd_keys[0], "CVD")
    lines.append(f"--- Trade Flow ({strategy_name}: {cvd_label}) ---")
    tp = data.get("tape", {})
    lines.append(f"  CVD: {_fmt(tp.get('tape.cvd'), 4)}")
    # Show only the strategy-specific CVD slopes/accels
    for key in cvd_keys:
        display_name = CVD_DISPLAY.get(key, key)
        lines.append(f"  {display_name}: {_fmt(tp.get(key), 4)}")
    lines.append(f"  Buy/Sell Ratio (60s): {_fmt(tp.get('tape.buy_sell_ratio_60s'))}")
    lines.append(
        f"  Flow Imbalance (60s): {_fmt(tp.get('tape.flow_imbalance_60s'))}  "
        f"(>0 = net buying)"
    )
    if full_micro:
        lines.append(
            f"  Large Trade Ratio (60s): "
            f"{_fmt(tp.get('tape.large_trade_ratio_60s'))}"
        )
    lines.append("")

    # ── Section 7: Multi-TF Trend ────────────────────────────────────
    lines.append("--- Multi-TF Trend ---")
    tr = data.get("trend", {})
    for tf in ["5m", "15m", "1h", "4h", "D"]:
        sep = tr.get(f"trend.ema_separation_{tf}")
        if sep is not None:
            bias = "bullish" if sep > 0 else "bearish" if sep < 0 else "neutral"
            marker = " *" if tf == primary_tf else ""
            lines.append(
                f"  [{tf:>3s}] EMA 20-50 Sep: {_fmt(sep)}%  ({bias}){marker}"
            )
    ema200_slope = tr.get("trend.ema200_slope_4h")
    if ema200_slope is not None:
        lines.append(f"  EMA 200 Slope (4h): {_fmt(ema200_slope)}%")
    ema50_200 = tr.get("trend.ema50_200_separation_4h")
    if ema50_200 is not None:
        lines.append(f"  EMA 50-200 Sep (4h): {_fmt(ema50_200)}%")
    lines.append("")

    # ── Section 8: Support / Resistance ──────────────────────────────
    lines.append("--- Support / Resistance ---")
    sr = data.get("support_resistance", {})
    pp = sr.get("sr.pivot_pp_distance_bps")
    if pp is not None:
        lines.append(f"  Pivot PP: {_fmt(pp)} bps from price")
        lines.append(
            f"  S1: {_fmt(sr.get('sr.pivot_s1_distance_bps'))} bps  "
            f"S2: {_fmt(sr.get('sr.pivot_s2_distance_bps'))} bps  "
            f"S3: {_fmt(sr.get('sr.pivot_s3_distance_bps'))} bps"
        )
        lines.append(
            f"  R1: {_fmt(sr.get('sr.pivot_r1_distance_bps'))} bps  "
            f"R2: {_fmt(sr.get('sr.pivot_r2_distance_bps'))} bps  "
            f"R3: {_fmt(sr.get('sr.pivot_r3_distance_bps'))} bps"
        )
    for tf in ["1h", "4h", "D"]:
        sw_r = sr.get(f"sr.swing_resistance_{tf}")
        sw_s = sr.get(f"sr.swing_support_{tf}")
        sw_r_pct = sr.get(f"sr.swing_resistance_pct_{tf}")
        sw_s_pct = sr.get(f"sr.swing_support_pct_{tf}")
        if sw_r is not None or sw_s is not None:
            lines.append(
                f"  [{tf}] Support: {_fmt(sw_s)} ({_fmt(sw_s_pct)}% away)  "
                f"Resistance: {_fmt(sw_r)} ({_fmt(sw_r_pct)}% away)"
            )
    lines.append("")

    # ── Section 9: Volume Profile ────────────────────────────────────
    lines.append("--- Volume Profile ---")
    vp = data.get("volume_profile", {})
    if vp:
        lines.append(f"  POC: {_fmt(vp.get('vp.poc_distance_bps'))} bps from price")
        lines.append(
            f"  VAH: {_fmt(vp.get('vp.vah_distance_bps'))} bps  "
            f"VAL: {_fmt(vp.get('vp.val_distance_bps'))} bps"
        )
        lines.append(
            f"  HVN: {_fmt(vp.get('vp.hvn_distance_bps'))} bps  "
            f"LVN: {_fmt(vp.get('vp.lvn_distance_bps'))} bps"
        )
        lines.append(
            f"  At Key Level: {'Yes' if vp.get('vp.at_level_flag') else 'No'}"
        )
    else:
        lines.append("  Insufficient data")
    lines.append("")

    # ── Section 10: Funding & OI ─────────────────────────────────────
    lines.append("--- Funding & Open Interest ---")
    fn = data.get("funding", {})
    lines.append(
        f"  Funding: {_fmt(fn.get('funding.current_bps'), 2)} bps  "
        f"(z-score: {_fmt(fn.get('funding.z_score'))})"
    )
    oid = data.get("oi", {})
    lines.append(
        f"  OI Delta 5m: {_fmt(oid.get('oi.delta_5m'))}  "
        f"Delta 1h: {_fmt(oid.get('oi.delta_1h'))}"
    )
    lines.append(
        f"  OI Trend 24h: {_fmt(oid.get('oi.trend_24h'))}  "
        f"Spike Z: {_fmt(oid.get('oi.spike_zscore'))}"
    )
    lines.append("")

    # ── Section 11: Market Structure (SMC) ───────────────────────────
    lines.append("--- Market Structure ---")
    st = data.get("structure", {})
    for tf in ["15m", "1h"]:
        tf_st = st.get(tf, {})
        if not tf_st:
            continue
        lines.append(f"  [{tf}]")
        sw_break = tf_st.get("structure.swing_break_pct")
        breakout = tf_st.get("structure.breakout_strength")
        sw_high = tf_st.get("structure.swing_high")
        sw_low = tf_st.get("structure.swing_low")
        momentum = tf_st.get("structure.swing_momentum")
        fvg_bull = tf_st.get("structure.fvg_bull")
        fvg_bear = tf_st.get("structure.fvg_bear")

        if sw_high is not None:
            lines.append(
                f"    Swing High: {_fmt(sw_high)}  Swing Low: {_fmt(sw_low)}"
            )
        lines.append(
            f"    Breakout Strength: {_fmt(breakout)}%  "
            f"Swing Break: {_fmt(sw_break)}%"
        )
        lines.append(f"    Momentum: {_fmt(momentum)}")
        if fvg_bull is not None:
            lines.append(f"    FVG Bull midpoint: {_fmt(fvg_bull)}")
        if fvg_bear is not None:
            lines.append(f"    FVG Bear midpoint: {_fmt(fvg_bear)}")
    lines.append("")

    # ── Section 12: Risk Gates (strategy thresholds shown) ───────────
    lines.append(f"--- Risk Gates ({strategy_name} thresholds) ---")
    rg = data.get("risk_gates", {})
    spread_val = rg.get("gate.spread_bps")
    spread_thresh = rg.get("gate.spread_threshold_bps")
    lines.append(
        f"  Spread: {_gate(rg.get('gate.spread_ok'))}  "
        f"({_fmt(spread_val)} bps"
        f"{f' < {_fmt(spread_thresh)} bps max' if spread_thresh else ''})"
    )
    depth_val = rg.get("gate.depth_usd")
    depth_thresh = rg.get("gate.depth_threshold")
    lines.append(
        f"  Liquidity: {_gate(rg.get('gate.liquidity_ok'))}  "
        f"({_usd(depth_val)}"
        f"{f' > {_usd(depth_thresh)} min' if depth_thresh else ''})"
    )
    chop_val = rg.get("gate.chop_index")
    chop_thresh = rg.get("gate.chop_threshold")
    lines.append(
        f"  Chop: {_gate(rg.get('gate.chop_ok'))}  "
        f"({_fmt(chop_val)}"
        f"{f' < {_fmt(chop_thresh)} max' if chop_thresh else ''})"
    )
    lines.append(f"  Data Quality: {_gate(rg.get('gate.data_quality_ok'))}")
    tradeable = rg.get("tradeable", False)
    lines.append(f"  Overall: {'TRADEABLE' if tradeable else 'NOT TRADEABLE'}")
    lines.append("")

    # ── Section 13: Strategy Constraints ─────────────────────────────
    lines.append(f"--- {strategy_name} Constraints ---")
    lines.append(
        f"  Min SL: {strat.get('min_sl_pct', '?')}%  "
        f"Min TP: {strat.get('min_tp_pct', '?')}%  "
        f"R:R: {strat.get('rr_floor', '?')}-{strat.get('rr_ceiling', '?')}"
    )
    lines.append(
        f"  Hold: {strat.get('hold_period', '?')}  "
        f"Primary ATR: atr.pct_{primary_tf}"
    )
    lines.append("")

    # ── Instrument ───────────────────────────────────────────────────
    inst = data.get("instrument", {})
    if inst:
        lines.append("--- Instrument ---")
        lines.append(
            f"  Tick: {inst.get('tick_size')}  Lot: {inst.get('lot_size')}  "
            f"Max Qty: {inst.get('max_order_qty')}  "
            f"Max Leverage: {inst.get('max_leverage')}x"
        )
        lines.append("")

    return "\n".join(lines)


# ── Quick Analysis ──────────────────────────────────────────────────


def format_quick(data: Dict[str, Any]) -> str:
    """Format a quick analysis dict to structured text."""
    lines: list[str] = []

    t = data.get("ticker", {})
    lines.append(f"=== {t.get('symbol', '???')} Quick Analysis ===")
    lines.append(
        f"Price: ${t.get('last_price', 0):,.2f}  "
        f"(Mark: ${t.get('mark_price', 0):,.2f})"
    )
    lines.append(f"24h Volume: {_usd(t.get('turnover_24h'))}")
    lines.append(
        f"Funding: {_fmt(t.get('funding_rate'), 6)}  |  "
        f"OI: {_usd(t.get('open_interest'))}"
    )
    lines.append("")

    # Microstructure
    ms = data.get("microstructure", {})
    lines.append("--- Orderbook ---")
    lines.append(f"  Microprice: ${_fmt(ms.get('ob.microprice'))}")
    lines.append(f"  Spread: {_fmt(ms.get('ob.spread_bps'))} bps")
    lines.append(f"  QI L1: {_fmt(ms.get('ob.qi_l1'))}")
    lines.append("")

    # 1h indicators
    ind_1h = (data.get("indicators") or {}).get("1h", {})
    if ind_1h and "error" not in ind_1h:
        lines.append("--- 1h Indicators ---")
        lines.append(f"  RSI: {_fmt(ind_1h.get('rsi.level_1h'))}")
        lines.append(f"  MACD Hist: {_fmt(ind_1h.get('macd.histogram_1h'), 4)}")
        lines.append(f"  BB%: {_fmt(ind_1h.get('boll.pct_1h'))}")
        lines.append(f"  ADX: {_fmt(ind_1h.get('adx.level_1h'))}")
        lines.append(f"  ATR%: {_fmt(ind_1h.get('atr.pct_1h'))}")
        lines.append(f"  Chop: {_fmt(ind_1h.get('chop.index_1h'))}")
        lines.append("")

    # Risk gates
    rg = data.get("risk_gates", {})
    lines.append("--- Risk Gates ---")
    tradeable = rg.get("tradeable", False)
    lines.append(
        f"  Spread: {_gate(rg.get('gate.spread_ok'))}  |  "
        f"Liquidity: {_gate(rg.get('gate.liquidity_ok'))}  |  "
        f"Chop: {_gate(rg.get('gate.chop_ok'))}  |  "
        f"Data: {_gate(rg.get('gate.data_quality_ok'))}"
    )
    lines.append(f"  Overall: {'TRADEABLE' if tradeable else 'NOT TRADEABLE'}")
    lines.append("")

    return "\n".join(lines)


# ── Account / Trading Formatters ────────────────────────────────────


def format_balance(data: Dict[str, Any]) -> str:
    """Format wallet balance response."""
    lines: list[str] = []

    lines.append("=== Wallet Balance ===")
    lines.append(f"  Equity: {_usd(data.get('total_equity'))}")
    lines.append(f"  Wallet Balance: {_usd(data.get('wallet_balance'))}")
    lines.append(f"  Available Balance: {_usd(data.get('available_balance'))}")
    lines.append(f"  Used Margin: {_usd(data.get('used_margin'))}")
    lines.append(f"  Unrealized PnL: {_usd(data.get('unrealized_pnl'))}")

    return "\n".join(lines)


def format_positions(positions: list) -> str:
    """Format open positions."""
    lines: list[str] = []
    lines.append("=== Open Positions ===")

    if not positions:
        lines.append("  No open positions.")
        return "\n".join(lines)

    for p in positions:
        size = float(p.get("size", 0))
        if size == 0:
            continue
        side = p.get("side", "?")
        symbol = p.get("symbol", "?")
        entry = float(p.get("avgPrice", 0))
        mark = float(p.get("markPrice", 0))
        upnl = float(p.get("unrealisedPnl", 0))
        leverage = p.get("leverage", "?")
        sl = p.get("stopLoss", "")
        tp = p.get("takeProfit", "")

        pnl_pct = ((mark - entry) / entry * 100) if entry > 0 else 0
        if side == "Sell":
            pnl_pct = -pnl_pct

        lines.append(f"  {symbol} {side.upper()} {size} @ {entry:.4f}")
        lines.append(
            f"    Mark: {mark:.4f}  PnL: {_usd(upnl)} ({pnl_pct:+.2f}%)"
        )
        lines.append(
            f"    Leverage: {leverage}x  SL: {sl or '—'}  TP: {tp or '—'}"
        )
        lines.append("")

    return "\n".join(lines)


def format_orders(orders: list) -> str:
    """Format open/conditional orders."""
    lines: list[str] = []
    lines.append("=== Open Orders ===")

    if not orders:
        lines.append("  No open orders.")
        return "\n".join(lines)

    for o in orders:
        symbol = o.get("symbol", "?")
        side = o.get("side", "?")
        otype = o.get("orderType", "?")
        qty = o.get("qty", "?")
        price = o.get("price", "?")
        status = o.get("orderStatus", "?")
        lines.append(
            f"  {symbol} {side} {otype} qty={qty} price={price} status={status}"
        )

    return "\n".join(lines)


def format_closed_pnl(records: list) -> str:
    """Format closed PnL records."""
    lines: list[str] = []
    lines.append("=== Closed PnL ===")

    if not records:
        lines.append("  No closed trades in period.")
        return "\n".join(lines)

    total_pnl = 0.0
    for r in records:
        symbol = r.get("symbol", "?")
        side = r.get("side", "?")
        closed_pnl = float(r.get("closedPnl", 0))
        total_pnl += closed_pnl
        entry = r.get("avgEntryPrice", "?")
        exit_ = r.get("avgExitPrice", "?")
        qty = r.get("qty", "?")
        created = r.get("createdTime", "?")
        lines.append(
            f"  {symbol} {side} qty={qty}  "
            f"entry={entry} exit={exit_}  "
            f"PnL: {_usd(closed_pnl)}  "
            f"at {created}"
        )

    lines.append("")
    lines.append(f"  Total: {_usd(total_pnl)}")
    return "\n".join(lines)
