"""
QuantClaw CLI — single entry point for all commands.

Usage: uv run quantclaw <command> [args]
"""

import argparse
import os
import sys
import time
from decimal import Decimal

from .bybit.client import BybitClient, BybitAPIError
from . import formatter


STRATEGY_CHOICES = ["fast", "swing", "medium", "position"]


def _get_client() -> BybitClient:
    """Create a BybitClient from environment variables."""
    api_key = os.environ.get("BYBIT_API_KEY")
    api_secret = os.environ.get("BYBIT_API_SECRET")
    mode = os.environ.get("BYBIT_MODE", "demo").lower()

    if not api_key or not api_secret:
        print("Error: BYBIT_API_KEY and BYBIT_API_SECRET must be set.", file=sys.stderr)
        sys.exit(1)

    return BybitClient(api_key=api_key, api_secret=api_secret, mode=mode)


def _get_mode() -> str:
    return os.environ.get("BYBIT_MODE", "demo").lower()


def _normalize_symbol(raw: str) -> str:
    """Normalise user input to SYMBOL format (e.g., 'btc' -> 'BTCUSDT')."""
    s = raw.upper().strip()
    if not s.endswith("USDT"):
        s += "USDT"
    return s


# ── Commands ────────────────────────────────────────────────────────


def cmd_analyze(args: argparse.Namespace) -> None:
    from .analysis import compiler

    client = _get_client()
    symbol = _normalize_symbol(args.symbol)

    if args.quick:
        data = compiler.compile_quick(symbol, client)
        print(formatter.format_quick(data))
    else:
        strategy = getattr(args, "strategy", "swing") or "swing"
        data = compiler.compile_full(symbol, client, strategy=strategy)
        print(formatter.format_full(data))


def cmd_balance(args: argparse.Namespace) -> None:
    client = _get_client()
    result = client.get_wallet_balance()
    print(formatter.format_balance(result))


def cmd_positions(args: argparse.Namespace) -> None:
    client = _get_client()
    positions = client.get_positions()
    print(formatter.format_positions(positions))


def cmd_pnl(args: argparse.Namespace) -> None:
    client = _get_client()
    days = args.days
    start_time = int((time.time() - days * 86400) * 1000)
    result = client.get_closed_pnl(limit=50, start_time=start_time)
    records = result.get("list", []) if isinstance(result, dict) else []
    print(formatter.format_closed_pnl(records))


def cmd_orders(args: argparse.Namespace) -> None:
    client = _get_client()
    symbol = _normalize_symbol(args.symbol) if args.symbol else None
    orders = client.get_open_orders(symbol=symbol)
    print(formatter.format_orders(orders))


def cmd_instrument(args: argparse.Namespace) -> None:
    client = _get_client()
    symbol = _normalize_symbol(args.symbol)
    inst = client.get_instrument_info(symbol)
    print(f"=== {symbol} Instrument Info ===")
    print(f"  Tick Size: {inst.tick_size}")
    print(f"  Lot Size: {inst.lot_size}")
    print(f"  Min Notional: {inst.min_notional}")
    print(f"  Max Order Qty: {inst.max_order_qty}")
    print(f"  Max Leverage: {inst.max_leverage}x")
    print(f"  Status: {inst.status}")


def cmd_buy(args: argparse.Namespace) -> None:
    _place_trade(args, side="Buy")


def cmd_sell(args: argparse.Namespace) -> None:
    _place_trade(args, side="Sell")


def _place_trade(args: argparse.Namespace, side: str) -> None:
    """Shared logic for buy (long) and sell (short) commands."""
    mode = _get_mode()
    client = _get_client()
    symbol = _normalize_symbol(args.symbol)
    size_usd = float(args.size_usd)
    leverage = int(args.leverage)
    sl_pct = args.sl
    tp_pct = args.tp
    strategy = getattr(args, "strategy", None)

    # Validate stop loss is provided
    if sl_pct is None:
        print("Error: --sl (stop loss %) is required.", file=sys.stderr)
        sys.exit(1)

    # Load strategy constraints if specified
    strat = None
    if strategy:
        from .analysis.strategies import get_strategy
        strat = get_strategy(strategy)

        # Validate SL against strategy minimum
        if sl_pct < strat["min_sl_pct"]:
            print(
                f"Error: SL {sl_pct}% is below {strategy.upper()} "
                f"minimum of {strat['min_sl_pct']}%.",
                file=sys.stderr,
            )
            sys.exit(1)

        # Validate TP against strategy minimum
        if tp_pct is not None and tp_pct < strat["min_tp_pct"]:
            print(
                f"Error: TP {tp_pct}% is below {strategy.upper()} "
                f"minimum of {strat['min_tp_pct']}%.",
                file=sys.stderr,
            )
            sys.exit(1)

    # Validate leverage
    if leverage > 5:
        print(
            f"Warning: Leverage {leverage}x exceeds default max of 5x. Proceeding as requested.",
            file=sys.stderr,
        )

    # Get instrument info for quantization
    inst = client.get_instrument_info(symbol)

    # Get current price
    ticker = client.get_ticker(symbol)
    mark_price = float(ticker.get("mark_price") or ticker.get("last_price", 0))
    if mark_price <= 0:
        print("Error: Could not get current price.", file=sys.stderr)
        sys.exit(1)

    # Calculate quantity
    raw_qty = Decimal(str(size_usd)) / Decimal(str(mark_price))
    qty = inst.quantize_qty(raw_qty)
    if qty <= 0:
        min_notional = float(inst.lot_size) * mark_price
        print(
            f"Error: Size ${size_usd} is below minimum order "
            f"(min lot {inst.lot_size} = ${min_notional:.2f}).",
            file=sys.stderr,
        )
        sys.exit(1)

    # Calculate SL price
    if side == "Buy":
        sl_price = inst.quantize_price(Decimal(str(mark_price * (1 - sl_pct / 100))))
    else:
        sl_price = inst.quantize_price(Decimal(str(mark_price * (1 + sl_pct / 100))))

    # Calculate TP price (optional)
    tp_price = None
    if tp_pct is not None:
        if side == "Buy":
            tp_price = inst.quantize_price(Decimal(str(mark_price * (1 + tp_pct / 100))))
        else:
            tp_price = inst.quantize_price(Decimal(str(mark_price * (1 - tp_pct / 100))))

    # Calculate R:R
    risk = abs(mark_price - float(sl_price))
    reward = abs(float(tp_price) - mark_price) if tp_price else None
    rr = round(reward / risk, 2) if reward and risk > 0 else None

    # Validate R:R against strategy bounds
    if strat and rr is not None:
        if rr < strat["rr_floor"]:
            print(
                f"Error: R:R {rr} is below {strategy.upper()} "
                f"floor of {strat['rr_floor']}.",
                file=sys.stderr,
            )
            sys.exit(1)
        if rr > strat["rr_ceiling"]:
            print(
                f"Error: R:R {rr} exceeds {strategy.upper()} "
                f"ceiling of {strat['rr_ceiling']}.",
                file=sys.stderr,
            )
            sys.exit(1)

    # Print summary
    if mode == "live":
        print("!!! LIVE MODE — REAL MONEY !!!")
        print()

    direction = "LONG" if side == "Buy" else "SHORT"
    print(f"=== Order Preview ===")
    print(f"  Mode: {mode.upper()}")
    if strat:
        print(f"  Strategy: {strategy.upper()}")
    print(f"  {symbol} {direction}")
    print(f"  Entry (mark): ${mark_price:,.4f}")
    print(f"  Qty: {qty} ({formatter._usd(size_usd)} notional)")
    print(f"  Leverage: {leverage}x")
    print(f"  Stop Loss: ${float(sl_price):,.4f} ({sl_pct}%)")
    if tp_price:
        print(f"  Take Profit: ${float(tp_price):,.4f} ({tp_pct}%)")
    if rr:
        print(f"  R:R Ratio: 1:{rr}")
    if strat:
        print(
            f"  Constraints: SL>={strat['min_sl_pct']}%  "
            f"TP>={strat['min_tp_pct']}%  "
            f"R:R {strat['rr_floor']}-{strat['rr_ceiling']}"
        )
    print()

    # Set leverage
    try:
        client.set_leverage(symbol, leverage)
    except BybitAPIError as e:
        # 110043 = leverage not modified (already set)
        if "110043" not in str(e):
            print(f"Warning: Failed to set leverage: {e}", file=sys.stderr)

    # Place order
    result = client.place_order(
        symbol=symbol,
        side=side,
        order_type="Market",
        qty=qty,
        stop_loss=sl_price,
        take_profit=tp_price,
    )

    order_id = result.get("result", {}).get("orderId", "?")
    print(f"=== Order Placed ===")
    print(f"  Order ID: {order_id}")
    print(f"  Mode: {mode.upper()}")


def cmd_close(args: argparse.Namespace) -> None:
    mode = _get_mode()
    client = _get_client()
    symbol = _normalize_symbol(args.symbol)

    if mode == "live":
        print("!!! LIVE MODE — REAL MONEY !!!")
        print()

    # Find open position
    positions = client.get_positions(symbol=symbol)
    open_pos = [p for p in positions if float(p.get("size", 0)) > 0]

    if not open_pos:
        print(f"No open position for {symbol}.")
        return

    pos = open_pos[0]
    pos_side = pos.get("side", "")
    size = pos.get("size", "0")
    close_side = "Sell" if pos_side == "Buy" else "Buy"

    print(f"Closing {symbol} {pos_side} position (size={size})...")

    result = client.place_order(
        symbol=symbol,
        side=close_side,
        order_type="Market",
        qty=Decimal(size),
        reduce_only=True,
    )

    order_id = result.get("result", {}).get("orderId", "?")
    print(f"=== Position Closed ===")
    print(f"  Order ID: {order_id}")
    print(f"  Mode: {mode.upper()}")


def cmd_set_leverage(args: argparse.Namespace) -> None:
    client = _get_client()
    symbol = _normalize_symbol(args.symbol)
    leverage = int(args.leverage)

    try:
        client.set_leverage(symbol, leverage)
        print(f"Leverage set to {leverage}x for {symbol}.")
    except BybitAPIError as e:
        if "110043" in str(e):
            print(f"Leverage already at {leverage}x for {symbol}.")
        else:
            raise


# ── Main ────────────────────────────────────────────────────────────


def main() -> None:
    parser = argparse.ArgumentParser(
        prog="quantclaw",
        description="QuantClaw — LLM-driven crypto perpetual futures trading on Bybit",
    )
    sub = parser.add_subparsers(dest="command", required=True)

    # analyze
    p_analyze = sub.add_parser("analyze", help="Market analysis")
    p_analyze.add_argument("symbol", help="Trading symbol (e.g., BTCUSDT or BTC)")
    p_analyze.add_argument("--quick", action="store_true", help="Quick analysis (ticker + OB + 1h only)")
    p_analyze.add_argument(
        "--strategy", choices=STRATEGY_CHOICES, default="swing",
        help="Strategy profile (default: swing)",
    )
    p_analyze.set_defaults(func=cmd_analyze)

    # balance
    p_balance = sub.add_parser("balance", help="Wallet balance")
    p_balance.set_defaults(func=cmd_balance)

    # positions
    p_positions = sub.add_parser("positions", help="Open positions")
    p_positions.set_defaults(func=cmd_positions)

    # pnl
    p_pnl = sub.add_parser("pnl", help="Closed PnL")
    p_pnl.add_argument("--days", type=int, default=7, help="Lookback days (default 7)")
    p_pnl.set_defaults(func=cmd_pnl)

    # orders
    p_orders = sub.add_parser("orders", help="Open/conditional orders")
    p_orders.add_argument("symbol", nargs="?", help="Filter by symbol")
    p_orders.set_defaults(func=cmd_orders)

    # instrument
    p_instrument = sub.add_parser("instrument", help="Instrument info")
    p_instrument.add_argument("symbol", help="Trading symbol")
    p_instrument.set_defaults(func=cmd_instrument)

    # buy
    p_buy = sub.add_parser("buy", help="Open long position")
    p_buy.add_argument("symbol", help="Trading symbol")
    p_buy.add_argument("size_usd", type=float, help="Position size in USD")
    p_buy.add_argument("--leverage", type=int, default=1, help="Leverage (default 1)")
    p_buy.add_argument("--sl", type=float, default=None, help="Stop loss %% from entry (REQUIRED)")
    p_buy.add_argument("--tp", type=float, default=None, help="Take profit %% from entry")
    p_buy.add_argument(
        "--strategy", choices=STRATEGY_CHOICES, default=None,
        help="Enforce strategy constraints (SL/TP/R:R bounds)",
    )
    p_buy.set_defaults(func=cmd_buy)

    # sell
    p_sell = sub.add_parser("sell", help="Open short position")
    p_sell.add_argument("symbol", help="Trading symbol")
    p_sell.add_argument("size_usd", type=float, help="Position size in USD")
    p_sell.add_argument("--leverage", type=int, default=1, help="Leverage (default 1)")
    p_sell.add_argument("--sl", type=float, default=None, help="Stop loss %% from entry (REQUIRED)")
    p_sell.add_argument("--tp", type=float, default=None, help="Take profit %% from entry")
    p_sell.add_argument(
        "--strategy", choices=STRATEGY_CHOICES, default=None,
        help="Enforce strategy constraints (SL/TP/R:R bounds)",
    )
    p_sell.set_defaults(func=cmd_sell)

    # close
    p_close = sub.add_parser("close", help="Close position")
    p_close.add_argument("symbol", help="Trading symbol")
    p_close.set_defaults(func=cmd_close)

    # set-leverage
    p_lev = sub.add_parser("set-leverage", help="Set leverage for symbol")
    p_lev.add_argument("symbol", help="Trading symbol")
    p_lev.add_argument("leverage", type=int, help="Leverage multiplier")
    p_lev.set_defaults(func=cmd_set_leverage)

    args = parser.parse_args()

    try:
        args.func(args)
    except BybitAPIError as e:
        print(f"Bybit API Error: {e}", file=sys.stderr)
        sys.exit(1)
    except KeyboardInterrupt:
        sys.exit(130)
    except Exception as e:
        print(f"Error: {e}", file=sys.stderr)
        sys.exit(1)
