"""
Bybit instrument model with quantisation helpers.
"""

from dataclasses import dataclass
from decimal import Decimal, ROUND_DOWN


@dataclass
class InstrumentInfo:
    """Trading instrument rules from Bybit.

    Used for order quantisation and validation.
    """

    symbol: str

    # Price / quantity rules
    tick_size: Decimal
    lot_size: Decimal
    min_notional: Decimal
    max_order_qty: Decimal

    # Leverage limits
    max_leverage: int

    # Trading status
    status: str = "Trading"

    def quantize_price(self, price: float) -> Decimal:
        """Round *price* down to the nearest tick size."""
        tick = self.tick_size
        return (Decimal(str(price)) / tick).quantize(
            Decimal("1"), rounding=ROUND_DOWN
        ) * tick

    def quantize_qty(self, qty: float) -> Decimal:
        """Round *qty* down to the nearest lot size."""
        lot = self.lot_size
        return (Decimal(str(qty)) / lot).quantize(
            Decimal("1"), rounding=ROUND_DOWN
        ) * lot
