"""
Bybit V5 API endpoint constants and interval mappings.
"""


class BybitEndpoints:
    """
    Bybit V5 API endpoints.

    REST base URLs:
      DEMO:  https://api-demo.bybit.com
      LIVE:  https://api.bybit.com
    """

    # REST API Base URLs
    REST_DEMO = "https://api-demo.bybit.com"
    REST_LIVE = "https://api.bybit.com"

    # Market Data
    KLINES = "/v5/market/kline"
    RECENT_TRADES = "/v5/market/recent-trade"
    ORDERBOOK = "/v5/market/orderbook"
    TICKERS = "/v5/market/tickers"
    INSTRUMENTS_INFO = "/v5/market/instruments-info"
    OPEN_INTEREST = "/v5/market/open-interest"
    FUNDING_HISTORY = "/v5/market/funding/history"
    SERVER_TIME = "/v5/market/time"

    # Trading
    PLACE_ORDER = "/v5/order/create"
    AMEND_ORDER = "/v5/order/amend"
    CANCEL_ORDER = "/v5/order/cancel"
    GET_OPEN_ORDERS = "/v5/order/realtime"
    GET_ORDER_HISTORY = "/v5/order/history"

    # Positions
    GET_POSITION_INFO = "/v5/position/list"
    GET_CLOSED_PNL = "/v5/position/closed-pnl"
    SET_LEVERAGE = "/v5/position/set-leverage"
    SET_TRADING_STOP = "/v5/position/trading-stop"

    # Account
    GET_WALLET_BALANCE = "/v5/account/wallet-balance"

    # Executions
    EXECUTION_LIST = "/v5/execution/list"

    @classmethod
    def get_rest_base_url(cls, mode: str) -> str:
        """Return the REST base URL for the given mode ('demo' or 'live')."""
        mode = mode.lower()
        if mode == "demo":
            return cls.REST_DEMO
        if mode == "live":
            return cls.REST_LIVE
        raise ValueError(
            f"Invalid BYBIT_MODE '{mode}'. Must be 'demo' or 'live'."
        )


# ---------------------------------------------------------------------------
# Interval mappings
# ---------------------------------------------------------------------------

INTERVAL_MAP = {
    "1m": "1",
    "3m": "3",
    "5m": "5",
    "15m": "15",
    "30m": "30",
    "1h": "60",
    "2h": "120",
    "4h": "240",
    "6h": "360",
    "12h": "720",
    "D": "D",
    "1d": "D",
    "W": "W",
    "1w": "W",
    "1M": "M",
}


def get_bybit_interval(timeframe: str) -> str:
    """Convert standard timeframe string to Bybit interval format.

    Raises ValueError for unsupported timeframes.
    """
    if timeframe not in INTERVAL_MAP:
        raise ValueError(f"Unsupported timeframe: {timeframe}")
    return INTERVAL_MAP[timeframe]


def get_standard_interval(bybit_interval: str) -> str:
    """Convert Bybit interval string back to standard timeframe.

    Raises ValueError for unrecognised intervals.
    """
    if bybit_interval == "1d":
        bybit_interval = "D"
    reverse_map = {v: k for k, v in INTERVAL_MAP.items() if k not in ("1d",)}
    if bybit_interval not in reverse_map:
        raise ValueError(f"Unrecognized Bybit interval: {bybit_interval}")
    return reverse_map[bybit_interval]
