"""
HMAC SHA256 authentication for Bybit V5 API.
"""

import hmac
import hashlib
import json
import time
from typing import Any, Dict
from urllib.parse import urlencode


def generate_signature(
    api_secret: str,
    api_key: str,
    timestamp: str,
    params: Dict[str, Any],
    recv_window: str = "5000",
    method: str = "POST",
) -> str:
    """
    Generate HMAC SHA256 signature for Bybit V5 API.

    V5 format varies by method:
    - GET:  timestamp + api_key + recv_window + query_string
    - POST: timestamp + api_key + recv_window + json_body

    Returns hex signature string.
    """
    if method == "GET":
        query_string = urlencode(sorted(params.items()))
        sign_string = timestamp + api_key + recv_window + query_string
    else:
        json_body = json.dumps(params, separators=(",", ":"))
        sign_string = timestamp + api_key + recv_window + json_body

    return hmac.new(
        api_secret.encode("utf-8"),
        sign_string.encode("utf-8"),
        hashlib.sha256,
    ).hexdigest()


def build_auth_headers(
    api_key: str,
    api_secret: str,
    params: Dict[str, Any],
    method: str = "POST",
    recv_window: str = "5000",
) -> Dict[str, str]:
    """
    Build authenticated request headers for Bybit V5 API.

    Uses current system time (milliseconds) for the timestamp.

    Returns dict of headers to merge into the request.
    """
    timestamp = str(int(time.time() * 1000))
    signature = generate_signature(
        api_secret, api_key, timestamp, params, recv_window, method
    )
    return {
        "X-BAPI-API-KEY": api_key,
        "X-BAPI-TIMESTAMP": timestamp,
        "X-BAPI-SIGN": signature,
        "X-BAPI-RECV-WINDOW": recv_window,
        "Content-Type": "application/json",
    }
