"""
Volume Profile Analysis

Calculates HVN, LVN, POC, Value Area High/Low with adaptive bucket resolution.
Ported from BebBot V3 volume_profile.py.
"""

import numpy as np
from typing import Dict, Any


def compute(
    closes: np.ndarray,
    volumes: np.ndarray,
    current_price: float,
    tick_size: float = 0.10,
    target_buckets: int = 100,
) -> Dict[str, Any]:
    """
    Calculate volume profile features.

    Args:
        closes: 1-D array of close prices (50+ for statistical validity).
        volumes: 1-D array of volumes corresponding to closes.
        current_price: Current market price.
        tick_size: Exchange tick size (for bucket alignment).
        target_buckets: Target number of buckets (adaptive based on range).

    Returns:
        Dict with HVN/LVN distances, POC, Value Area bounds (all in bps).
    """
    if len(closes) < 50 or len(volumes) < 50:
        return {
            "vp.hvn_distance_bps": None,
            "vp.lvn_distance_bps": None,
            "vp.at_level_flag": 0,
            "vp.poc_distance_bps": None,
            "vp.vah_distance_bps": None,
            "vp.val_distance_bps": None,
        }

    min_price = closes.min()
    max_price = closes.max()
    price_range = max_price - min_price

    # Prevent zero range (stale market or single price point)
    if price_range < tick_size:
        return {
            "vp.hvn_distance_bps": 0,
            "vp.lvn_distance_bps": 0,
            "vp.at_level_flag": 1,
            "vp.poc_distance_bps": 0,
            "vp.vah_distance_bps": 0,
            "vp.val_distance_bps": 0,
        }

    # Adaptive bucket sizing
    ideal_bucket_size = max(tick_size, price_range / target_buckets)

    # Align bucket size to tick multiples for clean boundaries
    ticks_per_bucket = max(1, round(ideal_bucket_size / tick_size))
    bucket_size = ticks_per_bucket * tick_size

    # Calculate actual number of buckets
    num_buckets = int(np.ceil(price_range / bucket_size))

    # Cap buckets at reasonable maximum
    if num_buckets > 500:
        num_buckets = 500
        bucket_size = price_range / num_buckets

    # Minimum buckets = 20 (fallback for very tight ranges)
    if num_buckets < 20:
        num_buckets = 20
        bucket_size = price_range / num_buckets

    # Aggregate volume by bucket
    volume_by_bucket = np.zeros(num_buckets)

    for price, volume in zip(closes, volumes):
        bucket_idx = int((price - min_price) / bucket_size)
        bucket_idx = min(bucket_idx, num_buckets - 1)
        volume_by_bucket[bucket_idx] += volume

    # Identify key levels
    hvn_idx = np.argmax(volume_by_bucket)  # Point of Control (POC)

    # LVN = lowest non-zero volume bucket
    non_zero_volume = volume_by_bucket[volume_by_bucket > 0]
    if len(non_zero_volume) > 0:
        lvn_vol = non_zero_volume.min()
        lvn_idx = np.where(volume_by_bucket == lvn_vol)[0][0]
    else:
        lvn_idx = 0

    # Calculate bucket midpoints
    hvn_price = min_price + (hvn_idx + 0.5) * bucket_size
    lvn_price = min_price + (lvn_idx + 0.5) * bucket_size
    poc_price = hvn_price  # POC = highest volume node

    # Calculate Value Area (70% of volume)
    total_volume = volume_by_bucket.sum()
    target_volume = total_volume * 0.70

    # Start from POC, expand up/down to capture 70% volume
    accumulated_volume = volume_by_bucket[hvn_idx]
    lower_idx = hvn_idx
    upper_idx = hvn_idx

    while accumulated_volume < target_volume and (
        lower_idx > 0 or upper_idx < num_buckets - 1
    ):
        lower_vol = volume_by_bucket[lower_idx - 1] if lower_idx > 0 else 0
        upper_vol = (
            volume_by_bucket[upper_idx + 1] if upper_idx < num_buckets - 1 else 0
        )

        if lower_vol > upper_vol and lower_idx > 0:
            lower_idx -= 1
            accumulated_volume += lower_vol
        elif upper_idx < num_buckets - 1:
            upper_idx += 1
            accumulated_volume += upper_vol
        else:
            break

    value_area_high = min_price + (upper_idx + 1) * bucket_size
    value_area_low = min_price + lower_idx * bucket_size

    # Check if current price is near key levels (within 1 bucket)
    current_bucket = int((current_price - min_price) / bucket_size)
    current_bucket = min(max(0, current_bucket), num_buckets - 1)
    at_hvn = abs(current_bucket - hvn_idx) <= 1
    at_lvn = abs(current_bucket - lvn_idx) <= 1
    at_level_flag = 1 if (at_hvn or at_lvn) else 0

    def _to_bps(level_price: float) -> float:
        return ((level_price - current_price) / current_price) * 10000

    return {
        "vp.hvn_distance_bps": round(_to_bps(hvn_price), 1),
        "vp.lvn_distance_bps": round(_to_bps(lvn_price), 1),
        "vp.at_level_flag": at_level_flag,
        "vp.poc_distance_bps": round(_to_bps(poc_price), 1),
        "vp.vah_distance_bps": round(_to_bps(value_area_high), 1),
        "vp.val_distance_bps": round(_to_bps(value_area_low), 1),
    }
