"""
QuantClaw Trend Analysis

Ported from BebBot V3 — battle-tested computations preserved exactly.
Pure functions: dict of klines in, dict of features out.
"""

import numpy as np
from typing import Dict, List, Any, Optional


def compute_trend(klines_dict: Dict[str, List[Dict[str, Any]]]) -> Dict[str, Any]:
    """
    Calculate trend features across multiple timeframes.

    Features:
    - EMA 20-50 separation per TF (raw %, positive=bullish, negative=bearish)
    - EMA 200 slope (4h)
    - EMA 50-200 separation (4h)

    Args:
        klines_dict: Dict mapping timeframe string to list of kline dicts.
                     Each kline dict has at least a "close" key.

    Returns:
        Dict with trend features.
    """
    features = {}

    # Calculate EMA separation (20-50) for each timeframe
    for tf in ["1m", "5m", "15m", "30m", "1h", "4h", "D", "W"]:
        klines = klines_dict.get(tf, [])

        if len(klines) >= 50:
            closes = np.array([float(k["close"]) for k in klines])

            ema20 = _ema(closes, 20)
            ema50 = _ema(closes, 50)

            if ema20 is not None and ema50 is not None and ema50 > 0:
                # % separation: (EMA20 - EMA50) / EMA50 * 100
                separation = ((ema20 - ema50) / ema50) * 100
            else:
                separation = None
        else:
            separation = None

        features[f"trend.ema_separation_{tf}"] = separation

    # EMA 200 slope (4h) — requires 200+ bars
    klines_4h = klines_dict.get("4h", [])

    if len(klines_4h) >= 210:
        closes_4h = np.array([float(k["close"]) for k in klines_4h])

        ema200_current = _ema(closes_4h, 200)
        ema200_10_bars_ago = _ema(closes_4h[:-10], 200)

        if ema200_current is not None and ema200_10_bars_ago is not None:
            slope = ((ema200_current - ema200_10_bars_ago) / ema200_10_bars_ago) * 100
        else:
            slope = None
    else:
        slope = None

    features["trend.ema200_slope_4h"] = slope

    # EMA 50-200 separation (4h)
    if len(klines_4h) >= 200:
        closes_4h = np.array([float(k["close"]) for k in klines_4h])

        ema50 = _ema(closes_4h, 50)
        ema200 = _ema(closes_4h, 200)

        if ema50 is not None and ema200 is not None and ema200 > 0:
            separation = ((ema50 - ema200) / ema200) * 100
        else:
            separation = None
    else:
        separation = None

    features["trend.ema50_200_separation_4h"] = separation

    return features


def _ema(data: np.ndarray, period: int) -> Optional[float]:
    """Calculate Exponential Moving Average."""
    if len(data) < period:
        return None

    alpha = 2 / (period + 1)
    ema = data[0]

    for price in data[1:]:
        ema = alpha * price + (1 - alpha) * ema

    return float(ema)
