"""
QuantClaw Support/Resistance Analysis

Ported from BebBot V3 — battle-tested computations preserved exactly.
Pure functions: klines dict + current price in, dict of features out.
"""

import numpy as np
from typing import Dict, List, Any


def compute_support_resistance(
    klines_dict: Dict[str, List[Dict[str, Any]]],
    current_price: float,
) -> Dict[str, Any]:
    """
    Calculate support and resistance levels.

    Args:
        klines_dict: Dict mapping timeframe to list of kline dicts.
                     Each kline has keys: high, low, close.
        current_price: Current market price.

    Returns:
        Dict with S/R features including pivot points.
    """
    # Guard against division by zero
    if not current_price or current_price <= 0:
        return {}

    features = {}

    # Calculate pivot points from 4h bars
    klines_4h = klines_dict.get("4h", [])
    if len(klines_4h) >= 6:
        # Get previous 6 bars (24 hours)
        lookback_bars = klines_4h[-12:-6] if len(klines_4h) >= 12 else klines_4h[-6:]
        prev_high = max(float(k["high"]) for k in lookback_bars)
        prev_low = min(float(k["low"]) for k in lookback_bars)
        prev_close = float(lookback_bars[-1]["close"])

        # Standard pivot point formula
        pp = (prev_high + prev_low + prev_close) / 3

        # Support levels
        s1 = (2 * pp) - prev_high
        s2 = pp - (prev_high - prev_low)
        s3 = prev_low - 2 * (prev_high - pp)

        # Resistance levels
        r1 = (2 * pp) - prev_low
        r2 = pp + (prev_high - prev_low)
        r3 = prev_high + 2 * (pp - prev_low)

        # Convert to distance in bps (positive = above price, negative = below)
        def to_bps(level: float) -> float:
            return round(((level - current_price) / current_price) * 10000, 1)

        features["sr.pivot_pp_distance_bps"] = to_bps(pp)
        features["sr.pivot_s1_distance_bps"] = to_bps(s1)
        features["sr.pivot_s2_distance_bps"] = to_bps(s2)
        features["sr.pivot_s3_distance_bps"] = to_bps(s3)
        features["sr.pivot_r1_distance_bps"] = to_bps(r1)
        features["sr.pivot_r2_distance_bps"] = to_bps(r2)
        features["sr.pivot_r3_distance_bps"] = to_bps(r3)

        # Confluence flag (near multiple levels within 50 bps)
        levels = [pp, s1, s2, s3, r1, r2, r3]
        near_levels = sum(1 for level in levels if abs(to_bps(level)) <= 50)
        features["sr.pivot_confluence"] = 1 if near_levels >= 2 else 0
    else:
        features["sr.pivot_pp_distance_bps"] = None
        features["sr.pivot_s1_distance_bps"] = None
        features["sr.pivot_s2_distance_bps"] = None
        features["sr.pivot_s3_distance_bps"] = None
        features["sr.pivot_r1_distance_bps"] = None
        features["sr.pivot_r2_distance_bps"] = None
        features["sr.pivot_r3_distance_bps"] = None
        features["sr.pivot_confluence"] = 0

    # Timeframe-specific lookback configurations
    # Format: {timeframe: (min_bars, lookback_bars, description)}
    tf_configs = {
        "1m": (20, 60, "1 hour"),
        "5m": (20, 60, "5 hours"),
        "15m": (20, 20, "5 hours"),
        "30m": (20, 20, "10 hours"),
        "1h": (20, 20, "20 hours"),
        "4h": (20, 12, "48 hours"),
        "D": (20, 20, "20 days"),
    }

    # Calculate swing highs/lows for all available timeframes
    for tf in ["1m", "5m", "15m", "30m", "1h", "4h", "D"]:
        klines = klines_dict.get(tf, [])

        min_bars, lookback_count, _description = tf_configs.get(tf, (20, 20, "default"))

        if len(klines) >= min_bars:
            # Apply timeframe-specific windowing
            windowed_klines = klines[-lookback_count:]

            highs = np.array([float(k["high"]) for k in windowed_klines])
            lows = np.array([float(k["low"]) for k in windowed_klines])

            # Find swing highs/lows (local extrema)
            swing_highs = _find_swing_points(highs, True)
            swing_lows = _find_swing_points(lows, False)

            # Swing resistance (above current price)
            resistances = [h for h in swing_highs if h > current_price]
            if resistances and current_price > 0:
                nearest_up = min(resistances)
                distance_up_pct = ((nearest_up - current_price) / current_price) * 100
            else:
                nearest_up = None
                distance_up_pct = None

            # Swing support (below current price)
            supports = [low_val for low_val in swing_lows if low_val < current_price]
            if supports and current_price > 0:
                nearest_down = max(supports)
                distance_down_pct = ((current_price - nearest_down) / current_price) * 100
            else:
                nearest_down = None
                distance_down_pct = None

            # Range-based SR (recent high/low — ALWAYS numeric)
            range_high = float(max(highs))
            range_low = float(min(lows))

            features[f"sr.swing_resistance_{tf}"] = nearest_up
            features[f"sr.swing_resistance_pct_{tf}"] = distance_up_pct
            features[f"sr.swing_support_{tf}"] = nearest_down
            features[f"sr.swing_support_pct_{tf}"] = distance_down_pct
            features[f"sr.range_high_{tf}"] = range_high
            features[f"sr.range_low_{tf}"] = range_low
            features[f"sr.range_high_pct_{tf}"] = ((range_high - current_price) / current_price) * 100
            features[f"sr.range_low_pct_{tf}"] = ((current_price - range_low) / current_price) * 100
        else:
            # Insufficient data
            features[f"sr.swing_resistance_{tf}"] = None
            features[f"sr.swing_resistance_pct_{tf}"] = None
            features[f"sr.swing_support_{tf}"] = None
            features[f"sr.swing_support_pct_{tf}"] = None
            features[f"sr.range_high_{tf}"] = None
            features[f"sr.range_low_{tf}"] = None
            features[f"sr.range_high_pct_{tf}"] = None
            features[f"sr.range_low_pct_{tf}"] = None

    return features


def _find_swing_points(data: np.ndarray, find_highs: bool) -> List[float]:
    """
    Find swing high/low points.

    Vectorized: rolling max/min over full window, then compare to center.
    """
    window = 3
    full_window = 2 * window + 1
    if len(data) < full_window:
        return []

    from pandas import Series

    s = Series(data)
    if find_highs:
        rolling_ext = s.rolling(full_window, center=True, min_periods=full_window).max()
        mask = s == rolling_ext
    else:
        rolling_ext = s.rolling(full_window, center=True, min_periods=full_window).min()
        mask = s == rolling_ext

    return [float(v) for v in s[mask].dropna().values]
