"""
Smart Money Concepts (SMC) Structure Analysis

Detects Break of Structure (BOS), Change of Character (CHoCH), and
Fair Value Gaps (FVG) using institutional swing detection methodology.
Ported from BebBot V3 structure_analysis.py.
"""

import numpy as np
from typing import Dict, Any, List, Optional, Tuple


def compute(
    highs: np.ndarray,
    lows: np.ndarray,
    closes: np.ndarray,
    lookback: int = 50,
    pivot_bars: int = 5,
) -> Dict[str, Any]:
    """
    Analyze market structure using SMC methodology.

    Detects:
    - Swing highs/lows (pivot points with n bars on each side)
    - BOS: Price breaks most recent swing high/low in trend direction
    - CHoCH: Price breaks swing in counter-trend direction
    - FVG: 3-candle pattern with gap (wick doesn't fill body gap)

    Args:
        highs: 1-D array of high prices.
        lows: 1-D array of low prices.
        closes: 1-D array of close prices.
        lookback: Bars to analyze (default 50).
        pivot_bars: Number of bars on each side for pivot detection.

    Returns:
        Dict with structure features.
    """
    _empty = {
        "structure.breakout_strength": None,
        "structure.swing_break_pct": None,
        "structure.swing_high": None,
        "structure.swing_low": None,
        "structure.fvg_bull": None,
        "structure.fvg_bear": None,
        "structure.swing_momentum": None,
    }

    if len(highs) < 20 or len(lows) < 20 or len(closes) < 20:
        return _empty

    # Limit to lookback period
    highs = highs[-lookback:]
    lows = lows[-lookback:]
    closes = closes[-lookback:]

    # Detect swing highs and lows
    swing_highs = _find_swing_highs(highs, left_bars=pivot_bars, right_bars=pivot_bars)
    swing_lows = _find_swing_lows(lows, left_bars=pivot_bars, right_bars=pivot_bars)

    if not swing_highs and not swing_lows:
        return _empty

    last_swing_high = swing_highs[-1] if swing_highs else None
    last_swing_low = swing_lows[-1] if swing_lows else None

    # Swing break percentage (continuous value)
    # Positive = bullish break, Negative = bearish break
    current_price = float(closes[-1])
    swing_break_pct = 0.0

    if last_swing_high and current_price > last_swing_high["price"]:
        swing_break_pct = (
            (current_price - last_swing_high["price"]) / last_swing_high["price"]
        ) * 100
    elif last_swing_low and current_price < last_swing_low["price"]:
        swing_break_pct = (
            (current_price - last_swing_low["price"]) / last_swing_low["price"]
        ) * 100

    # Swing momentum (continuous value)
    # Positive = higher lows (bullish), Negative = lower highs (bearish)
    swing_momentum = 0.0
    if len(swing_lows) >= 2:
        swing_momentum = (
            (swing_lows[-1]["price"] - swing_lows[-2]["price"])
            / swing_lows[-2]["price"]
        ) * 100
    elif len(swing_highs) >= 2:
        swing_momentum = (
            (swing_highs[-1]["price"] - swing_highs[-2]["price"])
            / swing_highs[-2]["price"]
        ) * 100

    # Breakout strength: % distance from recent range boundary
    recent_highs = highs[-20:]
    recent_lows = lows[-20:]
    range_high = float(np.max(recent_highs))
    range_low = float(np.min(recent_lows))
    range_midpoint = (range_high + range_low) / 2

    if current_price > range_high:
        breakout_strength = ((current_price - range_high) / range_high) * 100
    elif current_price < range_low:
        breakout_strength = ((current_price - range_low) / range_low) * 100
    else:
        if range_high != range_low:
            breakout_strength = (
                (current_price - range_midpoint) / (range_high - range_low)
            ) * 100
        else:
            breakout_strength = 0.0

    # Detect Fair Value Gaps (last 10 bars only)
    fvg_bull, fvg_bear = _detect_fvg(highs[-10:], lows[-10:])

    return {
        "structure.breakout_strength": float(breakout_strength),
        "structure.swing_break_pct": float(swing_break_pct),
        "structure.swing_high": last_swing_high["price"] if last_swing_high else None,
        "structure.swing_low": last_swing_low["price"] if last_swing_low else None,
        "structure.fvg_bull": fvg_bull,
        "structure.fvg_bear": fvg_bear,
        "structure.swing_momentum": float(swing_momentum),
    }


# ---------------------------------------------------------------------------
# Internal helpers
# ---------------------------------------------------------------------------


def _find_swing_highs(
    high: np.ndarray,
    left_bars: int = 5,
    right_bars: int = 5,
) -> List[Dict[str, Any]]:
    """Find swing highs (pivot highs with n bars on each side).

    Pure numpy: for each candidate index, check that it is strictly greater
    than all bars in the left and right windows.
    """
    n = len(high)
    window = left_bars + right_bars + 1
    if n < window:
        return []

    results: List[Dict[str, Any]] = []
    for i in range(left_bars, n - right_bars):
        left_max = np.max(high[i - left_bars : i])
        right_max = np.max(high[i + 1 : i + right_bars + 1])
        if high[i] > left_max and high[i] > right_max:
            results.append({"index": int(i), "price": float(high[i])})
    return results


def _find_swing_lows(
    low: np.ndarray,
    left_bars: int = 5,
    right_bars: int = 5,
) -> List[Dict[str, Any]]:
    """Find swing lows (pivot lows with n bars on each side).

    Pure numpy: for each candidate index, check that it is strictly less
    than all bars in the left and right windows.
    """
    n = len(low)
    window = left_bars + right_bars + 1
    if n < window:
        return []

    results: List[Dict[str, Any]] = []
    for i in range(left_bars, n - right_bars):
        left_min = np.min(low[i - left_bars : i])
        right_min = np.min(low[i + 1 : i + right_bars + 1])
        if low[i] < left_min and low[i] < right_min:
            results.append({"index": int(i), "price": float(low[i])})
    return results


def _detect_fvg(
    highs: np.ndarray,
    lows: np.ndarray,
) -> Tuple[Optional[float], Optional[float]]:
    """
    Detect Fair Value Gaps (3-candle pattern).

    Bullish FVG: candle[0].high < candle[2].low (gap not filled by candle[1])
    Bearish FVG: candle[0].low  > candle[2].high

    Returns (fvg_bull_midpoint, fvg_bear_midpoint) or None for each.
    """
    if len(highs) < 3:
        return None, None

    fvg_bull: Optional[float] = None
    fvg_bear: Optional[float] = None

    for i in range(len(highs) - 2):
        h0, h1, h2 = float(highs[i]), float(highs[i + 1]), float(highs[i + 2])
        l0, l1, l2 = float(lows[i]), float(lows[i + 1]), float(lows[i + 2])

        # Bullish FVG
        if h0 < l2 and h1 < l2:
            fvg_bull = (h0 + l2) / 2

        # Bearish FVG
        if l0 > h2 and l1 > h2:
            fvg_bear = (l0 + h2) / 2

    return fvg_bull, fvg_bear
