"""
Strategy profiles — single source of truth for all per-strategy constants.

Source: BebBot V3 feature_compiler_final.py, enums.py, constants.py, runtime_config.py
"""

from typing import Any, Dict


STRATEGIES: Dict[str, Dict[str, Any]] = {
    "fast": {
        "primary_tf": "5m",
        "lower_tf": "1m",
        "higher_tf": "1h",
        "cvd_keys": ["tape.cvd_30s_slope", "tape.cvd_accel_30s"],
        "spread_bps": 15,
        "depth_usd": 40_000,
        "chop_max": 61,
        "min_sl_pct": 0.5,
        "min_tp_pct": 0.8,
        "rr_floor": 1.2,
        "rr_ceiling": 2.0,
        "hold_period": "10-60 minutes",
        "fund_weight": ["sentiment", "events", "upcoming", "macro"],
        "full_microstructure": True,
    },
    "swing": {
        "primary_tf": "15m",
        "lower_tf": "5m",
        "higher_tf": "4h",
        "cvd_keys": ["tape.cvd_1h_slope"],
        "spread_bps": 25,
        "depth_usd": 50_000,
        "chop_max": 65,
        "min_sl_pct": 1.5,
        "min_tp_pct": 2.5,
        "rr_floor": 1.5,
        "rr_ceiling": 3.0,
        "hold_period": "2-8 hours",
        "fund_weight": ["sentiment", "events", "upcoming", "macro"],
        "full_microstructure": False,
    },
    "medium": {
        "primary_tf": "30m",
        "lower_tf": "15m",
        "higher_tf": "D",
        "cvd_keys": ["tape.cvd_1h_slope"],
        "spread_bps": 15,
        "depth_usd": 75_000,
        "chop_max": 60,
        "min_sl_pct": 0.8,
        "min_tp_pct": 1.0,
        "rr_floor": 1.5,
        "rr_ceiling": 3.0,
        "hold_period": "4-12 hours",
        "fund_weight": ["macro", "upcoming", "events", "sentiment"],
        "full_microstructure": False,
    },
    "position": {
        "primary_tf": "D",
        "lower_tf": "4h",
        "higher_tf": "W",
        "cvd_keys": ["tape.cvd_4h_slope"],
        "spread_bps": 40,
        "depth_usd": 120_000,
        "chop_max": 75,
        "min_sl_pct": 2.0,
        "min_tp_pct": 5.0,
        "rr_floor": 1.5,
        "rr_ceiling": 4.0,
        "hold_period": "1-4 weeks",
        "fund_weight": ["macro", "upcoming", "events", "sentiment"],
        "full_microstructure": False,
    },
}

ALL_TIMEFRAMES = ["1m", "5m", "15m", "30m", "1h", "4h", "D", "W"]

# CVD slope/accel keys subject to per-strategy filtering
CVD_FILTER_KEYS = {
    "tape.cvd_30s_slope", "tape.cvd_120s_slope", "tape.cvd_5m_slope",
    "tape.cvd_25m_slope", "tape.cvd_1h_slope", "tape.cvd_100m_slope",
    "tape.cvd_4h_slope", "tape.cvd_accel_30s", "tape.cvd_accel_2m",
    "tape.cvd_accel_5m",
}

# Display names for CVD keys
CVD_DISPLAY = {
    "tape.cvd_30s_slope": "CVD 30s Slope",
    "tape.cvd_1h_slope": "CVD 1h Slope",
    "tape.cvd_4h_slope": "CVD 4h Slope",
    "tape.cvd_accel_30s": "CVD 30s Accel",
}

# Short labels for trade flow section header
CVD_SHORT_LABEL = {
    "tape.cvd_30s_slope": "30s CVD",
    "tape.cvd_1h_slope": "1h CVD",
    "tape.cvd_4h_slope": "4h CVD",
    "tape.cvd_accel_30s": "30s CVD",
}


def get_strategy(name: str) -> Dict[str, Any]:
    """Get strategy config by name. Defaults to swing."""
    return STRATEGIES.get(name.lower(), STRATEGIES["swing"])
