"""
Open Interest Analysis

Calculates OI deltas, trends, and spike detection.
Ported from BebBot V3 oi_analysis.py.
"""

import numpy as np
from typing import Dict, Any, Optional


def compute(
    current_oi: Optional[float],
    oi_history: np.ndarray,
) -> Dict[str, Any]:
    """
    Calculate open interest features.

    Args:
        current_oi: Current open interest value.
        oi_history: 1-D array of historical OI values ordered oldest-to-newest.
            Assumes ~5-minute intervals between entries.

    Returns:
        Dict with OI delta, trend, and spike features.
    """
    if current_oi is None or len(oi_history) < 5:
        return {
            "oi.delta_5m": None,
            "oi.delta_1h": None,
            "oi.trend_24h": None,
            "oi.trend_72h": None,
            "oi.spike_zscore": None,
        }

    features: Dict[str, Any] = {}

    # OI delta 5min (most recent data point)
    if len(oi_history) >= 2:
        delta_5m = current_oi - oi_history[-2]
    else:
        delta_5m = 0.0

    features["oi.delta_5m"] = delta_5m

    # OI delta 1h (assuming 5-min intervals -> 12 entries = 1h)
    if len(oi_history) >= 12:
        delta_1h = current_oi - oi_history[-12]
    else:
        delta_1h = 0.0

    features["oi.delta_1h"] = delta_1h

    # OI trend 24h (linear regression slope over last 50 points)
    if len(oi_history) >= 10:
        oi_values = oi_history[-50:]
        x = np.arange(len(oi_values))

        if np.std(x) > 0:
            trend_24h = np.cov(x, oi_values)[0, 1] / np.var(x)
        else:
            trend_24h = 0.0
    else:
        trend_24h = 0.0

    features["oi.trend_24h"] = trend_24h

    # OI trend 72h (linear regression slope over full history)
    if len(oi_history) >= 30:
        x_72h = np.arange(len(oi_history))

        if np.std(x_72h) > 0:
            trend_72h = np.cov(x_72h, oi_history)[0, 1] / np.var(x_72h)
        else:
            trend_72h = 0.0
    else:
        trend_72h = None

    features["oi.trend_72h"] = trend_72h

    # OI spike z-score (continuous magnitude)
    if len(oi_history) >= 10:
        deltas = np.diff(oi_history)
        std_delta = np.std(deltas)

        spike_zscore = abs(delta_5m) / std_delta if std_delta > 0 else 0.0
    else:
        spike_zscore = None

    features["oi.spike_zscore"] = spike_zscore

    return features
