"""
QuantClaw Microstructure Analysis

Ported from BebBot V3 — battle-tested computations preserved exactly.
Pure functions: orderbook dict in, dict of features out.
Microprice history / drift tracking stripped (was cache-based in BebBot).
"""

import numpy as np
from typing import Dict, List, Any, Optional


def compute_microstructure(orderbook: Dict[str, Any]) -> Dict[str, Any]:
    """
    Calculate orderbook microstructure features from a single snapshot.

    Args:
        orderbook: Dict with keys: bids, asks
                   bids/asks are lists of [price, qty] pairs, best level first.

    Returns:
        Dict with microstructure features.
    """
    if not orderbook or not orderbook.get("bids") or not orderbook.get("asks"):
        return _empty_microstructure()

    bids = orderbook["bids"]
    asks = orderbook["asks"]

    features = {}

    # Best bid/ask
    bid_price, bid_qty = bids[0]
    ask_price, ask_qty = asks[0]

    # Microprice (volume-weighted mid)
    if bid_qty + ask_qty > 0:
        microprice = (bid_price * ask_qty + ask_price * bid_qty) / (bid_qty + ask_qty)
    else:
        microprice = (bid_price + ask_price) / 2

    features["ob.microprice"] = microprice

    # Spread (bps) — use mid price as denominator
    mid_price = (bid_price + ask_price) / 2
    if mid_price > 0:
        spread_bps = ((ask_price - bid_price) / mid_price) * 10000
    else:
        spread_bps = 0

    features["ob.spread_bps"] = spread_bps

    # Quote imbalance L1
    total_qty_l1 = bid_qty + ask_qty
    if total_qty_l1 > 0:
        imbalance_l1 = (bid_qty - ask_qty) / total_qty_l1
    else:
        imbalance_l1 = 0

    features["ob.qi_l1"] = imbalance_l1

    # Orderbook imbalance in basis points
    features["ob.imbalance_bps"] = imbalance_l1 * 10000

    # Quote imbalance L5
    bid_qty_l5 = sum(q for p, q in bids[:5])
    ask_qty_l5 = sum(q for p, q in asks[:5])
    total_qty_l5 = bid_qty_l5 + ask_qty_l5

    if total_qty_l5 > 0:
        imbalance_l5 = (bid_qty_l5 - ask_qty_l5) / total_qty_l5
    else:
        imbalance_l5 = 0

    features["ob.qi_l5"] = imbalance_l5

    # Depth USD L1
    bid_usd = bid_price * bid_qty
    ask_usd = ask_price * ask_qty
    total_usd = bid_usd + ask_usd
    features["ob.depth_usd_l1"] = total_usd

    # Depth USD L5
    depth_usd_l5_bid = sum(p * q for p, q in bids[:5])
    depth_usd_l5_ask = sum(p * q for p, q in asks[:5])
    features["ob.depth_usd_l5"] = depth_usd_l5_bid + depth_usd_l5_ask

    # Book slope L10 (regression of size by price level)
    if len(bids) >= 10 and len(asks) >= 10:
        bid_slope = _calculate_book_slope(bids[:10])
        ask_slope = _calculate_book_slope(asks[:10])
        features["ob.slope_bid_l10"] = bid_slope
        features["ob.slope_ask_l10"] = ask_slope
    else:
        features["ob.slope_bid_l10"] = 0
        features["ob.slope_ask_l10"] = 0

    return features


def _empty_microstructure() -> Dict[str, Any]:
    """Return dict with None/zero values for missing orderbook."""
    return {
        "ob.microprice": None,
        "ob.spread_bps": None,
        "ob.qi_l1": 0,
        "ob.qi_l5": 0,
        "ob.imbalance_bps": 0,
        "ob.depth_usd_l1": None,
        "ob.depth_usd_l5": None,
        "ob.slope_bid_l10": 0,
        "ob.slope_ask_l10": 0,
    }


def _calculate_book_slope(levels: List[List[float]]) -> float:
    """
    Calculate book slope via linear regression of qty by price.

    Positive slope: More liquidity at higher prices
    Negative slope: More liquidity at lower prices

    Args:
        levels: List of [price, qty] pairs

    Returns:
        Slope coefficient
    """
    if not levels or len(levels) < 2:
        return 0

    prices = np.array([p for p, q in levels])
    quantities = np.array([q for p, q in levels])

    # Linear regression: qty = slope * price + intercept
    if np.std(prices) == 0:
        return 0

    slope = np.cov(prices, quantities)[0, 1] / np.var(prices)

    return float(slope)
