"""
Funding Rate Analysis

Calculates funding rate features including z-score.
Ported from BebBot V3 funding_analysis.py.
"""

import numpy as np
from typing import Dict, Any, Optional


def compute(
    current_funding_rate: Optional[float],
    historical_rates: np.ndarray,
) -> Dict[str, Any]:
    """
    Calculate funding rate features.

    Args:
        current_funding_rate: Current funding rate as a decimal (e.g. 0.0001).
        historical_rates: 1-D array of historical funding rates as decimals.
            Needs >= 10 entries for z-score calculation.

    Returns:
        Dict with funding.current_bps and funding.z_score.
    """
    if current_funding_rate is None:
        return {
            "funding.current_bps": None,
            "funding.z_score": None,
        }

    # Convert to basis points
    funding_bps = current_funding_rate * 10000

    features: Dict[str, Any] = {"funding.current_bps": funding_bps}

    # Calculate z-score if we have enough history
    if len(historical_rates) >= 10:
        historical_bps = historical_rates * 10000
        mean = np.mean(historical_bps)
        std = np.std(historical_bps)

        if std > 0:
            z_score = (funding_bps - mean) / std
        else:
            z_score = 0.0
    else:
        z_score = 0.0

    features["funding.z_score"] = z_score

    return features
