"""
Feature compiler — stateless, synchronous orchestrator.

Fetches all market data from Bybit via REST and runs every analysis module
in a single pass. No cache, no async, no state between calls.
"""

import logging
from typing import Any, Dict, List, Optional

import numpy as np
import pandas as pd

from ..bybit.client import BybitClient
from . import (
    indicators,
    microstructure,
    tape,
    trend,
    support_resistance,
    volume_profile,
    funding,
    oi,
    structure,
    risk_gates,
)
from .strategies import get_strategy, CVD_FILTER_KEYS


logger = logging.getLogger(__name__)

TIMEFRAMES = ["1m", "5m", "15m", "30m", "1h", "4h", "D", "W"]
KLINE_LIMIT = 200


def _df_to_dict_list(df: pd.DataFrame) -> List[Dict[str, Any]]:
    """Convert a klines DataFrame to a list of dicts (for modules expecting dicts)."""
    if df.empty:
        return []
    return df.to_dict("records")


def _extract_arrays(df: pd.DataFrame):
    """Extract numpy arrays from a klines DataFrame."""
    if df.empty:
        return None, None, None, None
    return (
        df["high"].values,
        df["low"].values,
        df["close"].values,
        df["volume"].values if "volume" in df.columns else np.zeros(len(df)),
    )


def _compute_depth_usd(orderbook: Dict[str, Any]) -> float:
    """Compute total USD depth within 0.5% of mid price."""
    bids = orderbook.get("bids", [])
    asks = orderbook.get("asks", [])
    if not bids or not asks:
        return 0.0
    mid = (float(bids[0][0]) + float(asks[0][0])) / 2
    threshold = mid * 0.005
    total = 0.0
    for level in bids:
        price = float(level[0])
        if mid - price <= threshold:
            total += price * float(level[1])
    for level in asks:
        price = float(level[0])
        if price - mid <= threshold:
            total += price * float(level[1])
    return total


def compile_full(
    symbol: str,
    client: BybitClient,
    strategy: str = "swing",
) -> Dict[str, Any]:
    """
    Full market analysis: 14 REST calls, 200+ features.

    Args:
        symbol: Trading pair (e.g., "BTCUSDT").
        client: Authenticated BybitClient.
        strategy: Strategy profile ("fast", "swing", "medium", "position").

    Returns a dict organised by category:
        ticker, indicators, microstructure, tape, trend,
        support_resistance, volume_profile, funding, oi,
        structure, risk_gates, instrument, strategy
    """
    strat = get_strategy(strategy)

    # ── Fetch data (ticker is required, rest degrade gracefully) ─────
    ticker = client.get_ticker(symbol)

    orderbook: Dict[str, Any] = {"bids": [], "asks": []}
    trades: List[Dict[str, Any]] = []
    oi_data: List[Dict[str, Any]] = []
    funding_history: List[Dict[str, Any]] = []
    inst = None

    errors: List[str] = []

    try:
        orderbook = client.get_orderbook(symbol, limit=50)
    except Exception as e:
        errors.append(f"orderbook: {e}")
        logger.warning("Failed to fetch orderbook: %s", e)

    try:
        trades = client.get_recent_trades(symbol, limit=1000)
    except Exception as e:
        errors.append(f"trades: {e}")
        logger.warning("Failed to fetch trades: %s", e)

    try:
        oi_data = client.get_open_interest(symbol, interval="1h", limit=24)
    except Exception as e:
        errors.append(f"open_interest: {e}")
        logger.warning("Failed to fetch open interest: %s", e)

    try:
        funding_history = client.get_funding_history(symbol, limit=20)
    except Exception as e:
        errors.append(f"funding_history: {e}")
        logger.warning("Failed to fetch funding history: %s", e)

    try:
        inst = client.get_instrument_info(symbol)
    except Exception as e:
        errors.append(f"instrument: {e}")
        logger.warning("Failed to fetch instrument info: %s", e)

    klines_df: Dict[str, pd.DataFrame] = {}
    klines_dicts: Dict[str, List[Dict[str, Any]]] = {}
    for tf in TIMEFRAMES:
        try:
            df = client.get_klines(symbol, interval=tf, limit=KLINE_LIMIT)
        except Exception as e:
            errors.append(f"klines_{tf}: {e}")
            logger.warning("Failed to fetch %s klines: %s", tf, e)
            df = pd.DataFrame()
        klines_df[tf] = df
        klines_dicts[tf] = _df_to_dict_list(df)

    # ── Compute features ─────────────────────────────────────────────
    result: Dict[str, Any] = {}

    # Ticker basics (client returns snake_case keys)
    last_price = float(ticker.get("last_price", 0))
    result["ticker"] = {
        "symbol": symbol,
        "last_price": last_price,
        "volume_24h": float(ticker.get("volume_24h", 0)),
        "turnover_24h": float(ticker.get("turnover_24h", 0)),
        "funding_rate": float(ticker.get("funding_rate", 0)),
        "open_interest": float(ticker.get("open_interest", 0)),
        "mark_price": float(ticker.get("mark_price", 0)),
    }

    # Indicators per timeframe
    indicators_by_tf: Dict[str, Dict] = {}
    for tf in TIMEFRAMES:
        df = klines_df[tf]
        if len(df) >= 30:
            try:
                indicators_by_tf[tf] = indicators.compute_indicators(df, tf)
            except Exception as e:
                indicators_by_tf[tf] = {"error": str(e)}
        else:
            indicators_by_tf[tf] = {"error": f"insufficient data ({len(df)} bars)"}
    result["indicators"] = indicators_by_tf

    # Microstructure
    try:
        result["microstructure"] = microstructure.compute_microstructure(orderbook)
    except Exception as e:
        result["microstructure"] = {"error": str(e)}

    # Tape (trade flow)
    try:
        tape_data = tape.compute_tape(trades)
    except Exception as e:
        tape_data = {"error": str(e)}

    # Filter CVD keys per strategy
    if isinstance(tape_data, dict) and "error" not in tape_data:
        allowed_cvd = set(strat["cvd_keys"])
        tape_data = {
            k: v for k, v in tape_data.items()
            if k not in CVD_FILTER_KEYS or k in allowed_cvd
        }
    result["tape"] = tape_data

    # Multi-TF trend (expects dict of kline dict lists, NOT DataFrames)
    try:
        result["trend"] = trend.compute_trend(klines_dicts)
    except Exception as e:
        result["trend"] = {"error": str(e)}

    # Support / resistance (expects klines dict + price)
    try:
        result["support_resistance"] = support_resistance.compute_support_resistance(
            klines_dicts, last_price
        )
    except Exception as e:
        result["support_resistance"] = {"error": str(e)}

    # Volume profile (from 1h klines — needs closes and volumes arrays)
    df_1h = klines_df.get("1h", pd.DataFrame())
    if len(df_1h) >= 30:
        try:
            result["volume_profile"] = volume_profile.compute(
                closes=df_1h["close"].values,
                volumes=df_1h["volume"].values,
                current_price=last_price,
            )
        except Exception as e:
            result["volume_profile"] = {"error": str(e)}
    else:
        result["volume_profile"] = {}

    # Funding (expects current rate float + historical rates array)
    try:
        current_funding = float(ticker.get("funding_rate", 0))
        hist_rates = np.array([h["funding_rate"] for h in funding_history]) if funding_history else np.array([])
        result["funding"] = funding.compute(current_funding, hist_rates)
    except Exception as e:
        result["funding"] = {"error": str(e)}

    # Open interest (expects current OI float + historical OI array)
    try:
        current_oi = float(ticker.get("open_interest", 0))
        oi_hist = np.array([float(o.get("open_interest", 0)) for o in oi_data]) if oi_data else np.array([])
        result["oi"] = oi.compute(current_oi, oi_hist)
    except Exception as e:
        result["oi"] = {"error": str(e)}

    # Market structure (SMC) on 15m and 1h
    structure_results: Dict[str, Any] = {}
    for tf in ["15m", "1h"]:
        df = klines_df.get(tf, pd.DataFrame())
        if len(df) >= 20:
            try:
                highs, lows, closes, _ = _extract_arrays(df)
                structure_results[tf] = structure.compute(highs, lows, closes)
            except Exception as e:
                structure_results[tf] = {"error": str(e)}
    result["structure"] = structure_results

    # Risk gates (per-strategy thresholds)
    bids = orderbook.get("bids", [])
    asks = orderbook.get("asks", [])
    bid_price = float(bids[0][0]) if bids else None
    ask_price = float(asks[0][0]) if asks else None
    depth_usd = _compute_depth_usd(orderbook)
    is_btc = symbol.upper().startswith("BTC")

    # Get 1h highs/lows/closes for chop index calculation in risk gates
    df_1h_rg = klines_df.get("1h", pd.DataFrame())
    rg_highs, rg_lows, rg_closes, _ = _extract_arrays(df_1h_rg) if len(df_1h_rg) >= 14 else (None, None, None, None)

    try:
        result["risk_gates"] = risk_gates.compute(
            bid=bid_price,
            ask=ask_price,
            orderbook_depth_usd=depth_usd,
            highs=rg_highs,
            lows=rg_lows,
            closes=rg_closes,
            is_btc=is_btc,
            strategy=strategy,
        )
    except Exception as e:
        result["risk_gates"] = {"tradeable": False, "error": str(e)}

    # Instrument info
    if inst:
        result["instrument"] = {
            "tick_size": str(inst.tick_size),
            "lot_size": str(inst.lot_size),
            "max_order_qty": str(inst.max_order_qty),
            "max_leverage": str(inst.max_leverage),
        }
    else:
        result["instrument"] = {}

    # Strategy metadata (consumed by formatter)
    result["strategy"] = {
        "name": strategy,
        "primary_tf": strat["primary_tf"],
        "lower_tf": strat["lower_tf"],
        "higher_tf": strat["higher_tf"],
        "hold_period": strat["hold_period"],
        "min_sl_pct": strat["min_sl_pct"],
        "min_tp_pct": strat["min_tp_pct"],
        "rr_floor": strat["rr_floor"],
        "rr_ceiling": strat["rr_ceiling"],
        "fund_weight": strat["fund_weight"],
        "full_microstructure": strat["full_microstructure"],
        "cvd_keys": strat["cvd_keys"],
    }

    if errors:
        result["_fetch_errors"] = errors

    return result


def compile_quick(symbol: str, client: BybitClient) -> Dict[str, Any]:
    """
    Quick analysis: ticker + orderbook + 1h klines only.

    Returns a slimmed-down version of the full analysis.
    """
    ticker = client.get_ticker(symbol)
    orderbook = client.get_orderbook(symbol, limit=50)
    klines_1h = client.get_klines(symbol, interval="1h", limit=KLINE_LIMIT)

    last_price = float(ticker.get("last_price", 0))

    result: Dict[str, Any] = {}

    result["ticker"] = {
        "symbol": symbol,
        "last_price": last_price,
        "volume_24h": float(ticker.get("volume_24h", 0)),
        "turnover_24h": float(ticker.get("turnover_24h", 0)),
        "funding_rate": float(ticker.get("funding_rate", 0)),
        "open_interest": float(ticker.get("open_interest", 0)),
        "mark_price": float(ticker.get("mark_price", 0)),
    }

    result["microstructure"] = microstructure.compute_microstructure(orderbook)

    if len(klines_1h) >= 30:
        result["indicators"] = {"1h": indicators.compute_indicators(klines_1h, "1h")}
    else:
        result["indicators"] = {"1h": {"error": "insufficient data"}}

    # Risk gates (no strategy — uses legacy defaults)
    bids = orderbook.get("bids", [])
    asks = orderbook.get("asks", [])
    bid_price = float(bids[0][0]) if bids else None
    ask_price = float(asks[0][0]) if asks else None
    depth_usd = _compute_depth_usd(orderbook)
    is_btc = symbol.upper().startswith("BTC")

    rg_highs, rg_lows, rg_closes, _ = _extract_arrays(klines_1h) if len(klines_1h) >= 14 else (None, None, None, None)

    result["risk_gates"] = risk_gates.compute(
        bid=bid_price,
        ask=ask_price,
        orderbook_depth_usd=depth_usd,
        highs=rg_highs,
        lows=rg_lows,
        closes=rg_closes,
        is_btc=is_btc,
    )

    return result
