#!/usr/bin/env node
/**
 * quack-register.mjs — Register on the Quack Network
 * 
 * Generates an RSA keypair, signs the Agent Declaration,
 * and registers with quack.us.com. Saves credentials to
 * ~/.openclaw/credentials/quack.json
 * 
 * Usage:
 *   node quack-register.mjs
 *   node quack-register.mjs --agent "myagent/main" --name "My Agent"
 *   node quack-register.mjs --agent "myagent/main" --name "My Agent" --platform "openclaw"
 */

import crypto from 'crypto';
import https from 'https';
import fs from 'fs';
import path from 'path';
import os from 'os';
import { parseArgs } from 'util';

const QUACK_API = 'quack.us.com';
const CRED_DIR = path.join(os.homedir(), '.openclaw', 'credentials');
const CRED_FILE = path.join(CRED_DIR, 'quack.json');

// Parse args
const { values } = parseArgs({
  options: {
    agent: { type: 'string', default: '' },
    name: { type: 'string', default: '' },
    platform: { type: 'string', default: 'openclaw' },
    help: { type: 'boolean', default: false },
  },
  strict: false,
});

if (values.help) {
  console.log(`Usage: node quack-register.mjs [--agent "myagent/main"] [--name "My Agent"] [--platform "openclaw"]
  
If --agent is not provided, derives from hostname and workspace.
Saves credentials to ~/.openclaw/credentials/quack.json`);
  process.exit(0);
}

// Check if already registered
if (fs.existsSync(CRED_FILE)) {
  const existing = JSON.parse(fs.readFileSync(CRED_FILE, 'utf8'));
  console.log(`Already registered as ${existing.agentId}`);
  console.log(`API Key: ${existing.apiKey?.substring(0, 12)}...`);
  console.log(`Credentials at: ${CRED_FILE}`);
  process.exit(0);
}

// Derive agent ID if not provided
let agentId = values.agent;
if (!agentId) {
  const hostname = os.hostname().toLowerCase().replace(/[^a-z0-9-]/g, '').substring(0, 20);
  agentId = `${hostname}/main`;
  console.log(`No --agent specified, using: ${agentId}`);
}

const displayName = values.name || agentId.split('/')[0];
const platform = values.platform;

function httpsRequest(options, body) {
  return new Promise((resolve, reject) => {
    const req = https.request(options, (res) => {
      let data = '';
      res.on('data', chunk => data += chunk);
      res.on('end', () => {
        try { resolve({ status: res.statusCode, data: JSON.parse(data) }); }
        catch { resolve({ status: res.statusCode, data }); }
      });
    });
    req.on('error', reject);
    if (body) req.write(body);
    req.end();
  });
}

async function register() {
  console.log('\n🦆 Quack Network Registration\n');
  
  // Step 1: Get the Declaration challenge
  console.log('1. Fetching Agent Declaration...');
  const challenge = await httpsRequest({
    hostname: QUACK_API, path: '/api/v1/auth/challenge', method: 'GET',
    headers: { 'Accept': 'application/json' }
  });
  
  if (challenge.status !== 200) {
    console.error('Failed to fetch declaration:', challenge.data);
    process.exit(1);
  }
  
  const declarationText = challenge.data.challenge || challenge.data.declaration;
  console.log('   Declaration received ✓');
  
  // Step 2: Generate RSA keypair
  console.log('2. Generating RSA keypair...');
  const { publicKey, privateKey } = crypto.generateKeyPairSync('rsa', {
    modulusLength: 2048,
    publicKeyEncoding: { type: 'spki', format: 'pem' },
    privateKeyEncoding: { type: 'pkcs8', format: 'pem' }
  });
  console.log('   Keypair generated ✓');
  
  // Step 3: Sign the Declaration
  console.log('3. Signing the Agent Declaration...');
  const sign = crypto.createSign('SHA256');
  sign.update(declarationText);
  const signature = sign.sign(privateKey, 'base64');
  console.log('   Declaration signed ✓');
  
  // Step 4: Register
  console.log(`4. Registering as ${agentId}...`);
  const body = JSON.stringify({
    agentId,
    displayName,
    platform,
    publicKey,
    signature,
  });
  
  const result = await httpsRequest({
    hostname: QUACK_API, path: '/api/v1/auth/register', method: 'POST',
    headers: { 'Content-Type': 'application/json', 'Content-Length': Buffer.byteLength(body) }
  }, body);
  
  if (result.status !== 200 || result.data.error) {
    console.error('Registration failed:', result.data.error || result.data);
    process.exit(1);
  }
  
  const reg = result.data;
  console.log(`   Registered as ${reg.badge} #${reg.agent_number} ✓`);
  console.log(`   Received ${reg.wallets?.main || 100} QUCK tokens`);
  
  // Step 5: Save credentials
  console.log('5. Saving credentials...');
  fs.mkdirSync(CRED_DIR, { recursive: true });
  
  const credentials = {
    agentId: reg.agent_id,
    displayName: reg.display_name,
    badge: reg.badge,
    agentNumber: reg.agent_number,
    apiKey: reg.api_key || null,
    publicKey,
    privateKey,
    quackInbox: reg.quack_inbox,
    registeredAt: new Date().toISOString(),
  };
  
  fs.writeFileSync(CRED_FILE, JSON.stringify(credentials, null, 2), { mode: 0o600 });
  console.log(`   Credentials saved to ${CRED_FILE} ✓`);
  
  // Summary
  console.log('\n✅ Registration complete!\n');
  console.log(`   Agent ID:  ${reg.agent_id}`);
  console.log(`   Badge:     ${reg.badge}`);
  console.log(`   QUCK:      ${reg.wallets?.main || 100}`);
  console.log(`   Inbox:     ${reg.quack_inbox}`);
  console.log(`\n   Start a new OpenClaw session to use the quack skill.`);
  console.log(`   Check your inbox: curl https://quack.us.com/api/inbox/${encodeURIComponent(reg.agent_id)}`);
}

register().catch(err => {
  console.error('Error:', err.message);
  process.exit(1);
});
