---
name: qr-code-generator
description: "Use this skill when users need to create QR codes for any purpose. Triggers include: requests to \"generate QR code\", \"create QR\", \"make a QR code for\", or mentions of encoding data into scannable codes. Supports URLs, text, WiFi credentials, vCards (contact information), email addresses, phone numbers, SMS, location coordinates, calendar events, and custom data. Can customize colors, add logos, generate bulk QR codes, and export in multiple formats (PNG, SVG, PDF). Requires OpenClawCLI installation from clawhub.ai."
license: Proprietary
---

# QR Code Generator

Generate customizable QR codes for URLs, text, WiFi credentials, contact cards, and more. Supports batch generation, custom styling, logo embedding, and multiple export formats.

⚠️ **Prerequisite:** Install [OpenClawCLI](https://clawhub.ai/) (Windows, MacOS)

**Installation:**
```bash
# Standard installation
pip install qrcode[pil] segno

# If you encounter permission errors, use a virtual environment
python -m venv venv
source venv/bin/activate  # On Windows: venv\Scripts\activate
pip install qrcode[pil] segno
```

**Never use `--break-system-packages`** as it can damage your system's Python installation.

---

## Quick Reference

| Task | Command |
|------|---------|
| Basic URL QR code | `python scripts/qr.py "https://example.com"` |
| Text QR code | `python scripts/qr.py --type text "Hello World"` |
| WiFi QR code | `python scripts/qr.py --type wifi --ssid "MyNetwork" --password "secret"` |
| vCard contact | `python scripts/qr.py --type vcard --name "John Doe" --phone "+1234567890"` |
| Custom colors | `python scripts/qr.py "URL" --fg-color blue --bg-color white` |
| With logo | `python scripts/qr.py "URL" --logo logo.png` |
| SVG format | `python scripts/qr.py "URL" --format svg` |
| Batch generation | `python scripts/qr.py --batch urls.txt --output-dir qrcodes/` |

---

## Core Features

### 1. Multiple Data Types

Generate QR codes for various data types with automatic formatting.

**Supported Types:**
- **URL** - Websites and web links
- **Text** - Plain text messages
- **WiFi** - WiFi network credentials
- **vCard** - Contact information (VCF format)
- **Email** - Email addresses with optional subject/body
- **Phone** - Phone numbers (tel: links)
- **SMS** - SMS messages with recipient and text
- **Geo** - Geographic coordinates
- **Event** - Calendar events (iCal format)
- **Custom** - Any custom data

### 2. Customization Options

Personalize QR code appearance:
- Foreground and background colors
- Custom error correction levels
- Border size adjustment
- Module size control
- Logo/image embedding

### 3. Multiple Export Formats

Export in various formats for different use cases:
- **PNG** - Raster images (default)
- **SVG** - Vector graphics (scalable)
- **PDF** - Print-ready documents
- **EPS** - Vector format for design tools
- **Terminal** - ASCII art for terminal display

### 4. Batch Processing

Generate multiple QR codes from:
- Text files (one entry per line)
- CSV files (with metadata)
- JSON files (with configuration)

---

## Basic Usage

### URL QR Codes

Generate QR codes for websites and links.

```bash
# Simple URL
python scripts/qr.py "https://example.com"

# With custom filename
python scripts/qr.py "https://github.com" --output github_qr.png

# High error correction for printed codes
python scripts/qr.py "https://mysite.com" --error-correction H --output site_qr.png
```

**Output:**
```
QR code generated: qrcode.png
Size: 290x290 pixels
Error correction: M (Medium)
Data: https://example.com
```

### Text QR Codes

Encode plain text messages.

```bash
# Simple text
python scripts/qr.py --type text "Hello, World!"

# Multi-line text
python scripts/qr.py --type text "Line 1\nLine 2\nLine 3" --output message.png

# Large text (automatic size adjustment)
python scripts/qr.py --type text "$(cat message.txt)" --output text_qr.png
```

### WiFi QR Codes

Create scannable WiFi credentials.

```bash
# WPA/WPA2 network
python scripts/qr.py --type wifi --ssid "MyNetwork" --password "SecurePassword123"

# WPA2 network (explicit)
python scripts/qr.py --type wifi --ssid "HomeWiFi" --password "pass123" --security WPA

# Hidden network
python scripts/qr.py --type wifi --ssid "SecretNet" --password "secret" --hidden

# Open network (no password)
python scripts/qr.py --type wifi --ssid "GuestNetwork" --security nopass
```

**Security types:** `WPA`, `WEP`, `nopass`

**Output QR contains:**
```
WIFI:T:WPA;S:MyNetwork;P:SecurePassword123;H:false;;
```

### Contact Cards (vCard)

Generate vCard QR codes for easy contact sharing.

```bash
# Basic contact
python scripts/qr.py --type vcard --name "John Doe" --phone "+1234567890"

# Full contact details
python scripts/qr.py --type vcard \
  --name "Jane Smith" \
  --phone "+1234567890" \
  --email "jane@example.com" \
  --organization "Tech Corp" \
  --title "Senior Developer" \
  --url "https://janesmith.com" \
  --address "123 Main St, City, State, 12345" \
  --output jane_contact.png

# Multiple phone numbers
python scripts/qr.py --type vcard \
  --name "Bob Johnson" \
  --phone "+1234567890" \
  --phone-home "+0987654321" \
  --email "bob@email.com"
```

**Generated vCard format:**
```
BEGIN:VCARD
VERSION:3.0
FN:John Doe
TEL:+1234567890
END:VCARD
```

### Email QR Codes

Create mailto: links with optional subject and body.

```bash
# Simple email
python scripts/qr.py --type email --email "contact@example.com"

# With subject
python scripts/qr.py --type email --email "support@company.com" --subject "Support Request"

# With subject and body
python scripts/qr.py --type email \
  --email "info@example.com" \
  --subject "Inquiry" \
  --body "I would like more information about..."
```

**Output QR contains:**
```
mailto:contact@example.com?subject=Support%20Request&body=Message%20text
```

### Phone Number QR Codes

Generate clickable phone links.

```bash
# Simple phone number
python scripts/qr.py --type phone --phone "+1234567890"

# International format
python scripts/qr.py --type phone --phone "+44 20 7946 0958"
```

**Output QR contains:**
```
tel:+1234567890
```

### SMS QR Codes

Create pre-filled SMS messages.

```bash
# SMS with recipient only
python scripts/qr.py --type sms --phone "+1234567890"

# SMS with message
python scripts/qr.py --type sms --phone "+1234567890" --message "Hello from QR code!"
```

**Output QR contains:**
```
sms:+1234567890?body=Hello%20from%20QR%20code!
```

### Geographic Location QR Codes

Encode GPS coordinates.

```bash
# Coordinates only
python scripts/qr.py --type geo --latitude 37.7749 --longitude -122.4194

# With altitude
python scripts/qr.py --type geo --latitude 40.7128 --longitude -74.0060 --altitude 10

# Named location
python scripts/qr.py --type geo --latitude 51.5074 --longitude -0.1278 --location-name "London"
```

**Output QR contains:**
```
geo:37.7749,-122.4194
```

### Calendar Event QR Codes

Generate iCalendar event QR codes.

```bash
# Basic event
python scripts/qr.py --type event \
  --event-title "Team Meeting" \
  --event-start "2024-03-15T14:00:00" \
  --event-end "2024-03-15T15:00:00"

# Full event details
python scripts/qr.py --type event \
  --event-title "Conference 2024" \
  --event-start "2024-06-01T09:00:00" \
  --event-end "2024-06-01T17:00:00" \
  --event-location "Convention Center, NYC" \
  --event-description "Annual tech conference" \
  --output conference_qr.png
```

---

## Customization Options

### Colors

Customize foreground and background colors.

```bash
# Named colors
python scripts/qr.py "https://example.com" --fg-color blue --bg-color white

# Hex colors
python scripts/qr.py "https://example.com" --fg-color "#FF0000" --bg-color "#FFFFFF"

# RGB colors
python scripts/qr.py "https://example.com" --fg-color "rgb(0,100,200)" --bg-color "rgb(255,255,255)"

# Transparent background
python scripts/qr.py "https://example.com" --bg-color transparent --format png
```

**Common color names:** black, white, red, blue, green, yellow, orange, purple, pink, brown, gray

### Error Correction

Set error correction level (higher = more damage resistance).

```bash
--error-correction <L|M|Q|H>
```

**Levels:**
- **L** - Low (~7% recovery) - Use for digital display
- **M** - Medium (~15% recovery) - Default, good balance
- **Q** - Quartile (~25% recovery) - Use when adding logos
- **H** - High (~30% recovery) - Best for print, damaged surfaces

```bash
# Low (smallest QR code)
python scripts/qr.py "https://example.com" --error-correction L

# High (best for print with logo)
python scripts/qr.py "https://example.com" --error-correction H --logo company.png
```

### Size and Border

Control QR code size and border width.

```bash
# Custom module size (box size in pixels)
python scripts/qr.py "URL" --box-size 20

# Custom border (modules)
python scripts/qr.py "URL" --border 2

# Large QR code
python scripts/qr.py "URL" --box-size 30 --border 4 --output large_qr.png

# Minimal QR code (no border)
python scripts/qr.py "URL" --border 0 --output minimal_qr.png
```

**Defaults:**
- Box size: 10 pixels
- Border: 4 modules (recommended minimum)

### Logo Embedding

Add logos or images to QR codes.

```bash
# Add logo (center)
python scripts/qr.py "https://company.com" --logo company_logo.png

# Custom logo size (percentage of QR code)
python scripts/qr.py "URL" --logo logo.png --logo-size 20

# High error correction recommended with logos
python scripts/qr.py "URL" --logo logo.png --error-correction H
```

**Logo tips:**
- Use square or circular logos
- Keep logo size ≤ 30% of QR code
- Use high error correction (Q or H)
- Test scanning after adding logo

---

## Export Formats

### PNG (Default)

Raster image format, good for digital use.

```bash
python scripts/qr.py "https://example.com" --format png --output qr.png
```

**Best for:** Web, digital displays, simple sharing

### SVG (Vector)

Scalable vector graphics, perfect for any size.

```bash
python scripts/qr.py "https://example.com" --format svg --output qr.svg
```

**Best for:** Print, design work, scaling to any size

### PDF

Print-ready PDF documents.

```bash
python scripts/qr.py "https://example.com" --format pdf --output qr.pdf
```

**Best for:** Printing, documents, archival

### EPS

Encapsulated PostScript for professional design tools.

```bash
python scripts/qr.py "https://example.com" --format eps --output qr.eps
```

**Best for:** Professional design software (Adobe Illustrator, etc.)

### Terminal

Display QR code as ASCII art in terminal.

```bash
python scripts/qr.py "https://example.com" --format terminal
```

**Output:**
```
█████████████████████████████
█████████████████████████████
████ ▄▄▄▄▄ █▀█ █▄▄▄▄▄ ████
████ █   █ █▀▀▀█ █   █ ████
████ █▄▄▄█ █▀ ▀ █▄▄▄█ ████
...
```

**Best for:** Quick terminal display, debugging

---

## Batch Generation

### From Text File

Generate QR codes from a list of URLs or text.

```bash
# Create input file
cat > urls.txt << EOF
https://example.com
https://github.com
https://google.com
EOF

# Generate batch
python scripts/qr.py --batch urls.txt --output-dir qrcodes/
```

**Output:**
```
qrcodes/
  ├── qr_001.png
  ├── qr_002.png
  └── qr_003.png
```

### From CSV File

Generate with metadata (filenames, options).

```bash
# Create CSV
cat > contacts.csv << EOF
name,phone,email,filename
John Doe,+1234567890,john@example.com,john_qr.png
Jane Smith,+0987654321,jane@example.com,jane_qr.png
EOF

# Generate batch
python scripts/qr.py --batch contacts.csv --type vcard --output-dir contacts/
```

### From JSON File

Generate with full customization per QR code.

```bash
# Create JSON config
cat > qr_config.json << EOF
[
  {
    "data": "https://example.com",
    "output": "example_qr.png",
    "fg_color": "blue",
    "bg_color": "white"
  },
  {
    "type": "wifi",
    "ssid": "MyNetwork",
    "password": "secret",
    "output": "wifi_qr.png"
  }
]
EOF

# Generate batch
python scripts/qr.py --batch qr_config.json --output-dir custom/
```

---

## Common Workflows

### Event Check-In System

Generate QR codes for event tickets.

```bash
# Create ticket QR codes with unique IDs
python scripts/qr.py --type text "TICKET-001-VIP" --output tickets/ticket_001.png
python scripts/qr.py --type text "TICKET-002-GENERAL" --output tickets/ticket_002.png

# Or batch from CSV
cat > tickets.csv << EOF
ticket_id,type,name
TICKET-001,VIP,John Doe
TICKET-002,GENERAL,Jane Smith
EOF

python scripts/qr.py --batch tickets.csv --template "TICKET-{ticket_id}-{type}" --output-dir tickets/
```

### Restaurant Menu

Create QR code for digital menu.

```bash
# Menu URL
python scripts/qr.py "https://restaurant.com/menu" \
  --output menu_qr.png \
  --box-size 15 \
  --error-correction H

# Print version (PDF)
python scripts/qr.py "https://restaurant.com/menu" \
  --format pdf \
  --output menu_qr.pdf \
  --box-size 20
```

### WiFi Guest Access

Generate WiFi QR code for guests.

```bash
# Print-friendly version
python scripts/qr.py --type wifi \
  --ssid "Guest_Network" \
  --password "GuestPass123" \
  --format pdf \
  --output wifi_guest.pdf \
  --box-size 15 \
  --error-correction H

# Poster with logo
python scripts/qr.py --type wifi \
  --ssid "Guest_Network" \
  --password "GuestPass123" \
  --logo company_logo.png \
  --output wifi_poster.png \
  --box-size 20
```

### Contact Card Distribution

Create scannable business cards.

```bash
# Generate vCard
python scripts/qr.py --type vcard \
  --name "John Doe" \
  --phone "+1234567890" \
  --email "john@company.com" \
  --organization "Tech Corp" \
  --title "CEO" \
  --url "https://company.com" \
  --format svg \
  --output business_card.svg

# Print version
python scripts/qr.py --type vcard \
  --name "John Doe" \
  --phone "+1234567890" \
  --email "john@company.com" \
  --format pdf \
  --output business_card.pdf \
  --box-size 12
```

### Product Packaging

QR codes for product information.

```bash
# Product info URL
python scripts/qr.py "https://product.com/info/SKU12345" \
  --output product_qr.svg \
  --format svg \
  --fg-color "#000000" \
  --bg-color transparent

# With error correction for damaged packaging
python scripts/qr.py "https://product.com/info/SKU12345" \
  --error-correction H \
  --output product_qr_robust.png
```

### Social Media Links

QR codes for social profiles.

```bash
# Instagram
python scripts/qr.py "https://instagram.com/username" --output instagram_qr.png

# LinkedIn
python scripts/qr.py "https://linkedin.com/in/username" --output linkedin_qr.png

# Multiple platforms (batch)
cat > social.txt << EOF
https://twitter.com/username
https://instagram.com/username
https://linkedin.com/in/username
https://github.com/username
EOF

python scripts/qr.py --batch social.txt --output-dir social_qr/
```

### Payment Links

QR codes for payment services.

```bash
# PayPal
python scripts/qr.py "https://paypal.me/username" --output paypal_qr.png

# Venmo
python scripts/qr.py "venmo://username" --output venmo_qr.png

# Cash App
python scripts/qr.py "https://cash.app/$username" --output cashapp_qr.png
```

---

## Best Practices

### Size and Scanning

1. **Minimum size:** 2cm × 2cm for reliable scanning
2. **Viewing distance:** QR size should be 10% of scanning distance
3. **Border:** Keep at least 4 modules border (quiet zone)
4. **Testing:** Always test scan before printing

### Error Correction

1. **Digital display:** Use L or M
2. **Print without logo:** Use M
3. **Print with logo:** Use H
4. **Outdoor/damaged:** Use H

### Colors

1. **High contrast:** Dark foreground, light background
2. **Avoid:** Light colors on light, dark on dark
3. **Print:** Use pure black/white for best results
4. **Branding:** Test custom colors before mass production

### Logo Integration

1. **Size:** Keep logo ≤ 25-30% of QR code
2. **Error correction:** Use Q or H level
3. **Position:** Center placement works best
4. **Testing:** Verify scanning with logo

### File Formats

1. **Digital:** PNG for web, screens
2. **Print:** PDF or SVG for scalability
3. **Design:** SVG or EPS for editing
4. **Archive:** Keep source data and SVG version

---

## Troubleshooting

### Installation Issues

**"Missing required dependency"**
```bash
# Install dependencies
pip install qrcode[pil] segno

# Or use virtual environment
python -m venv venv
source venv/bin/activate
pip install qrcode[pil] segno
```

**"PIL/Pillow not found"**
```bash
pip install Pillow
```

### Generation Issues

**"QR code too complex"**
- Reduce data size
- Use higher version (auto-adjusts)
- Split into multiple QR codes

**"Cannot scan QR code"**
- Increase error correction
- Ensure sufficient contrast
- Check minimum size
- Remove or reduce logo size
- Test in good lighting

**"Logo obscures data"**
- Reduce logo size (--logo-size)
- Increase error correction to H
- Use simpler logo design

### File Issues

**"Cannot save file"**
- Check output directory exists
- Verify write permissions
- Check disk space

**"Invalid color format"**
- Use named colors: red, blue, green
- Use hex: #RRGGBB
- Use rgb: rgb(R,G,B)

---

## Command Reference

```bash
python scripts/qr.py [DATA] [OPTIONS]

DATA:
  Text string, URL, or data to encode (required unless using --batch)

GENERAL OPTIONS:
  --type              Data type (url|text|wifi|vcard|email|phone|sms|geo|event)
  -o, --output        Output filename (default: qrcode.png)
  -f, --format        Format (png|svg|pdf|eps|terminal)
  
CUSTOMIZATION:
  --fg-color          Foreground color (default: black)
  --bg-color          Background color (default: white)
  --error-correction  Error correction (L|M|Q|H, default: M)
  --box-size          Module size in pixels (default: 10)
  --border            Border size in modules (default: 4)
  --logo              Logo image path
  --logo-size         Logo size percentage (default: 20)

WIFI OPTIONS:
  --ssid              Network SSID
  --password          Network password
  --security          Security type (WPA|WEP|nopass)
  --hidden            Hidden network flag

VCARD OPTIONS:
  --name              Full name
  --phone             Phone number
  --phone-home        Home phone
  --phone-work        Work phone
  --email             Email address
  --organization      Company/organization
  --title             Job title
  --url               Website URL
  --address           Full address

EMAIL OPTIONS:
  --email             Email address
  --subject           Email subject
  --body              Email body

PHONE/SMS OPTIONS:
  --phone             Phone number
  --message           SMS message text

GEO OPTIONS:
  --latitude          Latitude coordinate
  --longitude         Longitude coordinate
  --altitude          Altitude (optional)
  --location-name     Location name (optional)

EVENT OPTIONS:
  --event-title       Event title
  --event-start       Start datetime (ISO format)
  --event-end         End datetime (ISO format)
  --event-location    Event location
  --event-description Event description

BATCH OPTIONS:
  --batch             Input file (txt|csv|json)
  --output-dir        Output directory for batch
  --template          Filename template for batch

HELP:
  --help              Show all options
```

---

## Examples by Use Case

### Quick QR Codes

```bash
# URL
python scripts/qr.py "https://example.com"

# Text
python scripts/qr.py --type text "Hello World"

# Phone
python scripts/qr.py --type phone --phone "+1234567890"
```

### Professional QR Codes

```bash
# With logo and custom colors
python scripts/qr.py "https://company.com" \
  --logo logo.png \
  --fg-color "#003366" \
  --bg-color "#FFFFFF" \
  --error-correction H \
  --output company_qr.png

# Print-ready
python scripts/qr.py "https://company.com" \
  --format pdf \
  --box-size 15 \
  --error-correction H \
  --output printable_qr.pdf
```

### Functional QR Codes

```bash
# WiFi access
python scripts/qr.py --type wifi --ssid "Network" --password "pass123"

# Contact card
python scripts/qr.py --type vcard --name "John Doe" --phone "+1234567890" --email "john@example.com"

# Calendar event
python scripts/qr.py --type event --event-title "Meeting" --event-start "2024-03-15T14:00:00" --event-end "2024-03-15T15:00:00"
```

### Bulk Generation

```bash
# From URL list
python scripts/qr.py --batch urls.txt --output-dir qrcodes/

# From CSV with metadata
python scripts/qr.py --batch data.csv --output-dir output/

# Custom configuration per code
python scripts/qr.py --batch config.json --output-dir custom/
```

---

## Support

For issues or questions:
1. Check this documentation
2. Run `python scripts/qr.py --help`
3. Verify dependencies are installed
4. Test with simple QR code first

**Resources:**
- OpenClawCLI: https://clawhub.ai/
- qrcode library: https://pypi.org/project/qrcode/
- segno library: https://pypi.org/project/segno/
- QR code specification: ISO/IEC 18004