#!/bin/bash
# Qlik Cloud Spaces Catalog
# List all spaces with details
#
# Usage: qlik-spaces.sh [limit]
#
# NOTE: Personal space is VIRTUAL in Qlik Cloud and will NOT appear here!
#       To list personal space apps, use: qlik-apps.sh --space personal

set -euo pipefail

LIMIT="${1:-100}"
TIMESTAMP=$(date -u +%Y-%m-%dT%H:%M:%SZ)

if [[ -z "${QLIK_TENANT:-}" ]] || [[ -z "${QLIK_API_KEY:-}" ]]; then
  echo "{\"success\":false,\"error\":\"QLIK_TENANT and QLIK_API_KEY required\",\"timestamp\":\"$TIMESTAMP\"}"
  exit 1
fi

TENANT="${QLIK_TENANT%/}"
[[ "$TENANT" != http* ]] && TENANT="https://$TENANT"

curl -sL \
  -H "Authorization: Bearer ${QLIK_API_KEY}" \
  -H "Content-Type: application/json" \
  "${TENANT}/api/v1/spaces?limit=${LIMIT}" | python3 -c "
import json
import sys

timestamp = '$TIMESTAMP'

try:
    data = json.load(sys.stdin)
    if 'errors' in data:
        print(json.dumps({'success': False, 'error': data['errors'][0].get('title', 'Unknown error'), 'timestamp': timestamp}, indent=2))
        sys.exit(1)
    
    spaces = []
    for s in data.get('data', []):
        spaces.append({
            'id': s.get('id'),
            'name': s.get('name'),
            'type': s.get('type'),
            'description': s.get('description'),
            'ownerId': s.get('ownerId')
        })
    
    # Count by type
    type_counts = {}
    for s in spaces:
        t = s.get('type', 'unknown')
        type_counts[t] = type_counts.get(t, 0) + 1
    
    print(json.dumps({
        'success': True, 
        'spaces': spaces, 
        'totalCount': len(spaces),
        'byType': type_counts,
        'note': 'Personal space is VIRTUAL and not listed here. Use qlik-apps.sh --space personal',
        'timestamp': timestamp
    }, indent=2))
except Exception as e:
    print(json.dumps({'success': False, 'error': str(e), 'timestamp': timestamp}, indent=2))
    sys.exit(1)
"
