#!/bin/bash
# Qlik Cloud Search
# Search for apps, datasets, automations, and more
# Usage: qlik-search.sh "query"

set -euo pipefail

QUERY="${1:-}"
TIMESTAMP=$(date -u +%Y-%m-%dT%H:%M:%SZ)

if [[ -z "${QLIK_TENANT:-}" ]] || [[ -z "${QLIK_API_KEY:-}" ]]; then
  echo "{\"success\":false,\"error\":\"QLIK_TENANT and QLIK_API_KEY required\",\"timestamp\":\"$TIMESTAMP\"}"
  exit 1
fi

if [[ -z "$QUERY" ]]; then
  echo "{\"success\":false,\"error\":\"Search query required. Usage: qlik-search.sh \\\"query\\\"\",\"timestamp\":\"$TIMESTAMP\"}"
  exit 1
fi

TENANT="${QLIK_TENANT%/}"
[[ "$TENANT" != http* ]] && TENANT="https://$TENANT"
ENCODED_QUERY=$(python3 -c "import urllib.parse; print(urllib.parse.quote('''$QUERY'''))")

curl -sL \
  -H "Authorization: Bearer ${QLIK_API_KEY}" \
  -H "Content-Type: application/json" \
  "${TENANT}/api/v1/items?query=${ENCODED_QUERY}&limit=50" | python3 -c "
import json
import sys

query = '''$QUERY'''
timestamp = '$TIMESTAMP'

try:
    data = json.load(sys.stdin)
    if 'errors' in data:
        print(json.dumps({'success': False, 'error': data['errors'][0].get('title', 'Unknown error'), 'timestamp': timestamp}, indent=2))
        sys.exit(1)
    
    results = []
    for item in data.get('data', []):
        results.append({
            'resourceId': item.get('resourceId'),
            'name': item.get('name'),
            'type': item.get('resourceType'),
            'description': (item.get('description') or '')[:100],
            'spaceId': item.get('spaceId'),
            'updated': item.get('updatedAt')
        })
    
    print(json.dumps({'success': True, 'query': query, 'results': results, 'totalCount': len(results), 'timestamp': timestamp}, indent=2))
except Exception as e:
    print(json.dumps({'success': False, 'error': str(e), 'timestamp': timestamp}, indent=2))
    sys.exit(1)
"
