#!/bin/bash
# Qlik Cloud Reload Trigger
# Trigger an app reload
# Usage: qlik-reload.sh <app-id>

set -euo pipefail

APP_ID="${1:-}"
TIMESTAMP=$(date -u +%Y-%m-%dT%H:%M:%SZ)

if [[ -z "${QLIK_TENANT:-}" ]] || [[ -z "${QLIK_API_KEY:-}" ]]; then
  echo "{\"success\":false,\"error\":\"QLIK_TENANT and QLIK_API_KEY required\",\"timestamp\":\"$TIMESTAMP\"}"
  exit 1
fi

if [[ -z "$APP_ID" ]]; then
  echo "{\"success\":false,\"error\":\"App ID required. Usage: qlik-reload.sh <app-id>\",\"timestamp\":\"$TIMESTAMP\"}"
  exit 1
fi

TENANT="${QLIK_TENANT%/}"
[[ "$TENANT" != http* ]] && TENANT="https://$TENANT"

curl -sL -X POST \
  -H "Authorization: Bearer ${QLIK_API_KEY}" \
  -H "Content-Type: application/json" \
  -d "{\"appId\": \"${APP_ID}\"}" \
  "${TENANT}/api/v1/reloads" | python3 -c "
import json
import sys

app_id = '$APP_ID'
timestamp = '$TIMESTAMP'

try:
    data = json.load(sys.stdin)
    
    if 'errors' in data:
        error = data['errors'][0].get('title', data['errors'][0].get('detail', 'Unknown error'))
        print(json.dumps({'success': False, 'error': error, 'appId': app_id, 'timestamp': timestamp}, indent=2))
        sys.exit(1)
    
    result = {
        'success': True,
        'reload': {
            'id': data.get('id'),
            'appId': data.get('appId'),
            'status': data.get('status'),
            'type': data.get('type'),
            'creationTime': data.get('creationTime')
        },
        'message': 'Reload triggered successfully',
        'timestamp': timestamp
    }
    print(json.dumps(result, indent=2))
except Exception as e:
    print(json.dumps({'success': False, 'error': str(e), 'timestamp': timestamp}, indent=2))
    sys.exit(1)
"
