#!/bin/bash
# Qlik Cloud Duplicate Apps Finder
# Find apps with duplicate names
# Usage: qlik-duplicates.sh [min-count]
#
# min-count: Minimum duplicates to report (default: 2)

set -euo pipefail

MIN_COUNT="${1:-2}"
TIMESTAMP=$(date -u +%Y-%m-%dT%H:%M:%SZ)

if [[ -z "${QLIK_TENANT:-}" ]] || [[ -z "${QLIK_API_KEY:-}" ]]; then
  echo "{\"success\":false,\"error\":\"QLIK_TENANT and QLIK_API_KEY required\",\"timestamp\":\"$TIMESTAMP\"}"
  exit 1
fi

TENANT="${QLIK_TENANT%/}"
[[ "$TENANT" != http* ]] && TENANT="https://$TENANT"

curl -sL \
  -H "Authorization: Bearer ${QLIK_API_KEY}" \
  -H "Content-Type: application/json" \
  "${TENANT}/api/v1/apps?limit=500" | python3 -c "
import json
import sys
from collections import defaultdict

min_count = int('$MIN_COUNT')
timestamp = '$TIMESTAMP'

try:
    data = json.load(sys.stdin)
    apps = data.get('data', [])
    
    # Group apps by name
    by_name = defaultdict(list)
    for app in apps:
        attrs = app.get('attributes', {})
        name = attrs.get('name', 'Unnamed')
        by_name[name].append({
            'id': attrs.get('id'),
            'name': name,
            'spaceId': attrs.get('spaceId'),
            'ownerId': attrs.get('ownerId'),
            'modified': attrs.get('modifiedDate'),
            'lastReload': attrs.get('lastReloadTime')
        })
    
    # Find duplicates
    duplicates = []
    for name, apps_list in by_name.items():
        if len(apps_list) >= min_count:
            # Sort by modified date (newest first)
            apps_list.sort(key=lambda x: x.get('modified') or '', reverse=True)
            duplicates.append({
                'name': name,
                'count': len(apps_list),
                'apps': apps_list
            })
    
    # Sort by count descending
    duplicates.sort(key=lambda x: -x['count'])
    
    total_wasted = sum(d['count'] - 1 for d in duplicates)
    
    result = {
        'success': True,
        'totalApps': len(apps),
        'uniqueNames': len(by_name),
        'duplicateNames': len(duplicates),
        'wastedApps': total_wasted,
        'duplicates': duplicates[:20],  # Top 20
        'timestamp': timestamp
    }
    
    if len(duplicates) > 20:
        result['note'] = f'Showing top 20 of {len(duplicates)} duplicate groups'
    
    print(json.dumps(result, indent=2))
except Exception as e:
    print(json.dumps({'success': False, 'error': str(e), 'timestamp': timestamp}, indent=2))
    sys.exit(1)
"
