#!/bin/bash
# Qlik Cloud Datasets (Managed Data)
# List managed datasets (requires Data Integration license)
# Usage: qlik-datasets.sh [space-id] [limit]
#
# Note: For uploaded files, use qlik-datafiles.sh instead.

set -euo pipefail

SPACE_ID="${1:-}"
LIMIT="${2:-50}"
TIMESTAMP=$(date -u +%Y-%m-%dT%H:%M:%SZ)

if [[ -z "${QLIK_TENANT:-}" ]] || [[ -z "${QLIK_API_KEY:-}" ]]; then
  echo "{\"success\":false,\"error\":\"QLIK_TENANT and QLIK_API_KEY required\",\"timestamp\":\"$TIMESTAMP\"}"
  exit 1
fi

TENANT="${QLIK_TENANT%/}"
[[ "$TENANT" != http* ]] && TENANT="https://$TENANT"

PARAMS="limit=${LIMIT}"
if [[ -n "$SPACE_ID" ]]; then
  PARAMS="${PARAMS}&spaceId=${SPACE_ID}"
fi

curl -sL \
  -H "Authorization: Bearer ${QLIK_API_KEY}" \
  -H "Content-Type: application/json" \
  "${TENANT}/api/v1/data-sets?${PARAMS}" | python3 -c "
import json
import sys

timestamp = '$TIMESTAMP'

try:
    data = json.load(sys.stdin)
    
    if 'errors' in data:
        print(json.dumps({'success': False, 'error': data['errors'][0].get('title', 'Unknown error'), 'timestamp': timestamp}, indent=2))
        sys.exit(1)
    
    datasets = []
    for d in data.get('data', []):
        op = d.get('operational', {})
        datasets.append({
            'id': d.get('id'),
            'name': d.get('name'),
            'description': d.get('description'),
            'type': d.get('type', op.get('logicalType')),
            'spaceId': d.get('spaceId'),
            'ownerId': d.get('ownerId'),
            'size': op.get('size') or op.get('sizeBytes'),
            'rowCount': op.get('rowCount') or op.get('recordCount'),
            'columnCount': op.get('columnCount') or op.get('fieldCount'),
            'secureQri': d.get('secureQri'),
            'createdAt': d.get('createdAt'),
            'updatedAt': d.get('updatedAt')
        })
    
    print(json.dumps({
        'success': True,
        'datasets': datasets,
        'totalCount': len(datasets),
        'timestamp': timestamp
    }, indent=2))
except Exception as e:
    print(json.dumps({'success': False, 'error': str(e), 'timestamp': timestamp}, indent=2))
    sys.exit(1)
"
