#!/bin/bash
# Qlik Cloud Data Files
# List uploaded data files
# Usage: qlik-datafiles.sh [space-id] [limit]

set -euo pipefail

SPACE_ID="${1:-}"
LIMIT="${2:-50}"
TIMESTAMP=$(date -u +%Y-%m-%dT%H:%M:%SZ)

if [[ -z "${QLIK_TENANT:-}" ]] || [[ -z "${QLIK_API_KEY:-}" ]]; then
  echo "{\"success\":false,\"error\":\"QLIK_TENANT and QLIK_API_KEY required\",\"timestamp\":\"$TIMESTAMP\"}"
  exit 1
fi

TENANT="${QLIK_TENANT%/}"
[[ "$TENANT" != http* ]] && TENANT="https://$TENANT"

PARAMS="limit=${LIMIT}"
if [[ -n "$SPACE_ID" ]]; then
  PARAMS="${PARAMS}&spaceId=${SPACE_ID}"
fi

curl -sL \
  -H "Authorization: Bearer ${QLIK_API_KEY}" \
  -H "Content-Type: application/json" \
  "${TENANT}/api/v1/data-files?${PARAMS}" | python3 -c "
import json
import sys

timestamp = '$TIMESTAMP'

try:
    data = json.load(sys.stdin)
    
    if 'errors' in data:
        print(json.dumps({'success': False, 'error': data['errors'][0].get('title', 'Unknown error'), 'timestamp': timestamp}, indent=2))
        sys.exit(1)
    
    files = []
    for f in data.get('data', []):
        files.append({
            'id': f.get('id'),
            'name': f.get('name'),
            'folder': f.get('folder', False),
            'size': f.get('size'),
            'spaceId': f.get('spaceId'),
            'ownerId': f.get('ownerId'),
            'qri': f.get('qri'),
            'createdDate': f.get('createdDate'),
            'modifiedDate': f.get('modifiedDate')
        })
    
    print(json.dumps({
        'success': True,
        'dataFiles': files,
        'totalCount': len(files),
        'timestamp': timestamp
    }, indent=2))
except Exception as e:
    print(json.dumps({'success': False, 'error': str(e), 'timestamp': timestamp}, indent=2))
    sys.exit(1)
"
