#!/bin/bash
# Qlik Cloud Automations
# List automations
# Usage: qlik-automations.sh [limit]

set -euo pipefail

LIMIT="${1:-50}"
TIMESTAMP=$(date -u +%Y-%m-%dT%H:%M:%SZ)

if [[ -z "${QLIK_TENANT:-}" ]] || [[ -z "${QLIK_API_KEY:-}" ]]; then
  echo "{\"success\":false,\"error\":\"QLIK_TENANT and QLIK_API_KEY required\",\"timestamp\":\"$TIMESTAMP\"}"
  exit 1
fi

TENANT="${QLIK_TENANT%/}"
[[ "$TENANT" != http* ]] && TENANT="https://$TENANT"

curl -sL \
  -H "Authorization: Bearer ${QLIK_API_KEY}" \
  -H "Content-Type: application/json" \
  "${TENANT}/api/v1/automations?limit=${LIMIT}" | python3 -c "
import json
import sys

timestamp = '$TIMESTAMP'

try:
    data = json.load(sys.stdin)
    
    if 'errors' in data:
        print(json.dumps({'success': False, 'error': data['errors'][0].get('title', 'Unknown error'), 'timestamp': timestamp}, indent=2))
        sys.exit(1)
    
    automations = []
    for a in data.get('data', []):
        automations.append({
            'id': a.get('id'),
            'name': a.get('name'),
            'description': a.get('description'),
            'state': a.get('state'),
            'lastRunStatus': a.get('lastRunStatus'),
            'lastRunTime': a.get('lastRunTime'),
            'ownerId': a.get('ownerId'),
            'created': a.get('createdAt'),
            'updated': a.get('updatedAt')
        })
    
    print(json.dumps({
        'success': True,
        'automations': automations,
        'totalCount': len(automations),
        'timestamp': timestamp
    }, indent=2))
except Exception as e:
    print(json.dumps({'success': False, 'error': str(e), 'timestamp': timestamp}, indent=2))
    sys.exit(1)
"
