#!/bin/bash
# Qlik Cloud Automation Runs
# Get run history for an automation
# Usage: qlik-automation-runs.sh <automation-id> [limit]

set -euo pipefail

AUTOMATION_ID="${1:-}"
LIMIT="${2:-10}"
TIMESTAMP=$(date -u +%Y-%m-%dT%H:%M:%SZ)

if [[ -z "${QLIK_TENANT:-}" ]] || [[ -z "${QLIK_API_KEY:-}" ]]; then
  echo "{\"success\":false,\"error\":\"QLIK_TENANT and QLIK_API_KEY required\",\"timestamp\":\"$TIMESTAMP\"}"
  exit 1
fi

if [[ -z "$AUTOMATION_ID" ]]; then
  echo "{\"success\":false,\"error\":\"Automation ID required. Usage: qlik-automation-runs.sh <automation-id> [limit]\",\"timestamp\":\"$TIMESTAMP\"}"
  exit 1
fi

TENANT="${QLIK_TENANT%/}"
[[ "$TENANT" != http* ]] && TENANT="https://$TENANT"

curl -sL \
  -H "Authorization: Bearer ${QLIK_API_KEY}" \
  -H "Content-Type: application/json" \
  "${TENANT}/api/v1/automations/${AUTOMATION_ID}/runs?limit=${LIMIT}" | python3 -c "
import json
import sys

automation_id = '$AUTOMATION_ID'
timestamp = '$TIMESTAMP'

try:
    data = json.load(sys.stdin)
    
    if 'errors' in data:
        print(json.dumps({'success': False, 'error': data['errors'][0].get('title', 'Unknown error'), 'timestamp': timestamp}, indent=2))
        sys.exit(1)
    
    runs = []
    for r in data.get('data', data.get('runs', [])):
        runs.append({
            'id': r.get('id'),
            'status': r.get('status'),
            'startTime': r.get('startTime'),
            'stopTime': r.get('stopTime'),
            'context': r.get('context')
        })
    
    print(json.dumps({
        'success': True,
        'automationId': automation_id,
        'runs': runs,
        'count': len(runs),
        'timestamp': timestamp
    }, indent=2))
except Exception as e:
    print(json.dumps({'success': False, 'error': str(e), 'timestamp': timestamp}, indent=2))
    sys.exit(1)
"
