#!/bin/bash
# Qlik Cloud Automation Details
# Get details of a specific automation
# Usage: qlik-automation-get.sh <automation-id>

set -euo pipefail

AUTOMATION_ID="${1:-}"
TIMESTAMP=$(date -u +%Y-%m-%dT%H:%M:%SZ)

if [[ -z "${QLIK_TENANT:-}" ]] || [[ -z "${QLIK_API_KEY:-}" ]]; then
  echo "{\"success\":false,\"error\":\"QLIK_TENANT and QLIK_API_KEY required\",\"timestamp\":\"$TIMESTAMP\"}"
  exit 1
fi

if [[ -z "$AUTOMATION_ID" ]]; then
  echo "{\"success\":false,\"error\":\"Automation ID required. Usage: qlik-automation-get.sh <automation-id>\",\"timestamp\":\"$TIMESTAMP\"}"
  exit 1
fi

TENANT="${QLIK_TENANT%/}"
[[ "$TENANT" != http* ]] && TENANT="https://$TENANT"

curl -sL \
  -H "Authorization: Bearer ${QLIK_API_KEY}" \
  -H "Content-Type: application/json" \
  "${TENANT}/api/v1/automations/${AUTOMATION_ID}" | python3 -c "
import json
import sys

timestamp = '$TIMESTAMP'

try:
    data = json.load(sys.stdin)
    if 'errors' in data:
        print(json.dumps({'success': False, 'error': data['errors'][0].get('title', 'Unknown error'), 'timestamp': timestamp}, indent=2))
        sys.exit(1)
    
    # Extract key fields
    automation = {
        'id': data.get('id'),
        'name': data.get('name'),
        'description': data.get('description'),
        'state': data.get('state'),
        'ownerId': data.get('ownerId'),
        'lastRunStatus': data.get('lastRunStatus'),
        'lastRunTime': data.get('lastRunTime'),
        'created': data.get('createdAt'),
        'updated': data.get('updatedAt')
    }
    
    print(json.dumps({'success': True, 'automation': automation, 'timestamp': timestamp}, indent=2))
except Exception as e:
    print(json.dumps({'success': False, 'error': str(e), 'timestamp': timestamp}, indent=2))
    sys.exit(1)
"
