#!/bin/bash
# Qlik Cloud Get App
# Get details of a specific app
# Usage: qlik-app-get.sh <app-id>

set -euo pipefail

APP_ID="${1:-}"
TIMESTAMP=$(date -u +%Y-%m-%dT%H:%M:%SZ)

if [[ -z "${QLIK_TENANT:-}" ]] || [[ -z "${QLIK_API_KEY:-}" ]]; then
  echo "{\"success\":false,\"error\":\"QLIK_TENANT and QLIK_API_KEY required\",\"timestamp\":\"$TIMESTAMP\"}"
  exit 1
fi

if [[ -z "$APP_ID" ]]; then
  echo "{\"success\":false,\"error\":\"App ID required. Usage: qlik-app-get.sh <app-id>\",\"timestamp\":\"$TIMESTAMP\"}"
  exit 1
fi

TENANT="${QLIK_TENANT%/}"
[[ "$TENANT" != http* ]] && TENANT="https://$TENANT"

curl -sL \
  -H "Authorization: Bearer ${QLIK_API_KEY}" \
  -H "Content-Type: application/json" \
  "${TENANT}/api/v1/apps/${APP_ID}" | python3 -c "
import json
import sys

timestamp = '$TIMESTAMP'

try:
    data = json.load(sys.stdin)
    
    if 'errors' in data:
        print(json.dumps({'success': False, 'error': data['errors'][0].get('title', 'Unknown error'), 'timestamp': timestamp}, indent=2))
        sys.exit(1)
    
    attrs = data.get('attributes', data)
    result = {
        'success': True,
        'app': {
            'id': data.get('id', attrs.get('id')),
            'name': attrs.get('name'),
            'description': attrs.get('description'),
            'spaceId': attrs.get('spaceId'),
            'ownerId': attrs.get('ownerId'),
            'published': attrs.get('published', False),
            'publishTime': attrs.get('publishTime'),
            'lastReloadTime': attrs.get('lastReloadTime'),
            'thumbnail': attrs.get('thumbnail'),
            'created': attrs.get('createdDate'),
            'modified': attrs.get('modifiedDate')
        },
        'timestamp': timestamp
    }
    print(json.dumps(result, indent=2))
except Exception as e:
    print(json.dumps({'success': False, 'error': str(e), 'timestamp': timestamp}, indent=2))
    sys.exit(1)
"
