#!/bin/bash
# Qlik Cloud Delete App
# Delete an app
# Usage: qlik-app-delete.sh <app-id>

set -euo pipefail

APP_ID="${1:-}"
TIMESTAMP=$(date -u +%Y-%m-%dT%H:%M:%SZ)

if [[ -z "${QLIK_TENANT:-}" ]] || [[ -z "${QLIK_API_KEY:-}" ]]; then
  echo "{\"success\":false,\"error\":\"QLIK_TENANT and QLIK_API_KEY required\",\"timestamp\":\"$TIMESTAMP\"}"
  exit 1
fi

if [[ -z "$APP_ID" ]]; then
  echo "{\"success\":false,\"error\":\"App ID required. Usage: qlik-app-delete.sh <app-id>\",\"timestamp\":\"$TIMESTAMP\"}"
  exit 1
fi

TENANT="${QLIK_TENANT%/}"
[[ "$TENANT" != http* ]] && TENANT="https://$TENANT"

HTTP_CODE=$(curl -sL -w "%{http_code}" -o /tmp/qlik_delete_response.txt -X DELETE \
  -H "Authorization: Bearer ${QLIK_API_KEY}" \
  -H "Content-Type: application/json" \
  "${TENANT}/api/v1/apps/${APP_ID}")

RESPONSE=$(cat /tmp/qlik_delete_response.txt 2>/dev/null || echo "")

python3 -c "
import json
import sys

app_id = '$APP_ID'
http_code = '$HTTP_CODE'
response = '''$RESPONSE'''
timestamp = '$TIMESTAMP'

try:
    if http_code == '204' or (http_code == '200' and not response.strip()):
        print(json.dumps({'success': True, 'appId': app_id, 'message': 'App deleted successfully', 'timestamp': timestamp}, indent=2))
        sys.exit(0)
    
    if response.strip():
        data = json.loads(response)
        if 'errors' in data:
            print(json.dumps({'success': False, 'error': data['errors'][0].get('title', 'Unknown error'), 'appId': app_id, 'timestamp': timestamp}, indent=2))
            sys.exit(1)
    
    print(json.dumps({'success': False, 'error': f'Unexpected response: HTTP {http_code}', 'appId': app_id, 'timestamp': timestamp}, indent=2))
    sys.exit(1)
except json.JSONDecodeError:
    if http_code.startswith('2'):
        print(json.dumps({'success': True, 'appId': app_id, 'message': 'App deleted successfully', 'timestamp': timestamp}, indent=2))
    else:
        print(json.dumps({'success': False, 'error': f'HTTP {http_code}: {response[:200]}', 'timestamp': timestamp}, indent=2))
        sys.exit(1)
except Exception as e:
    print(json.dumps({'success': False, 'error': str(e), 'timestamp': timestamp}, indent=2))
    sys.exit(1)
"
