#!/bin/bash
# Qlik Cloud Answers - List Assistants
# List Qlik Answers AI assistants (Cloud-only)
# Usage: qlik-answers-assistants.sh [limit]

set -euo pipefail

LIMIT="${1:-50}"
TIMESTAMP=$(date -u +%Y-%m-%dT%H:%M:%SZ)

if [[ -z "${QLIK_TENANT:-}" ]] || [[ -z "${QLIK_API_KEY:-}" ]]; then
  echo "{\"success\":false,\"error\":\"QLIK_TENANT and QLIK_API_KEY required\",\"timestamp\":\"$TIMESTAMP\"}"
  exit 1
fi

TENANT="${QLIK_TENANT%/}"
[[ "$TENANT" != http* ]] && TENANT="https://$TENANT"

curl -sL \
  -H "Authorization: Bearer ${QLIK_API_KEY}" \
  -H "Content-Type: application/json" \
  "${TENANT}/api/v1/assistants?limit=${LIMIT}" | python3 -c "
import json
import sys

timestamp = '$TIMESTAMP'

try:
    raw = sys.stdin.read()
    if not raw.strip() or '<html>' in raw.lower():
        print(json.dumps({'success': True, 'assistants': [], 'totalCount': 0, 'note': 'Qlik Answers may not be enabled on this tenant', 'timestamp': timestamp}, indent=2))
        sys.exit(0)
    
    data = json.loads(raw)
    if 'errors' in data:
        print(json.dumps({'success': False, 'error': data['errors'][0].get('title', 'Unknown error'), 'timestamp': timestamp}, indent=2))
        sys.exit(1)
    
    assistants = []
    for a in data.get('data', []):
        assistants.append({
            'id': a.get('id'),
            'name': a.get('name'),
            'title': a.get('title'),
            'description': a.get('description'),
            'spaceId': a.get('spaceId'),
            'ownerId': a.get('ownerId'),
            'knowledgeBases': a.get('knowledgeBases', []),
            'created': a.get('createdAt'),
            'updated': a.get('updatedAt')
        })
    
    print(json.dumps({'success': True, 'assistants': assistants, 'totalCount': len(assistants), 'timestamp': timestamp}, indent=2))
except json.JSONDecodeError:
    print(json.dumps({'success': True, 'assistants': [], 'totalCount': 0, 'note': 'Qlik Answers not available or not enabled', 'timestamp': timestamp}, indent=2))
except Exception as e:
    print(json.dumps({'success': False, 'error': str(e), 'timestamp': timestamp}, indent=2))
    sys.exit(1)
"
