#!/bin/bash
# Qlik Cloud Answers - Ask Question
# Ask a question to a Qlik Answers AI assistant (Cloud-only)
# Usage: qlik-answers-ask.sh <assistant-id> "question" [thread-id]
#
# If no thread-id provided, creates a new thread automatically.

set -euo pipefail

ASSISTANT_ID="${1:-}"
QUESTION="${2:-}"
THREAD_ID="${3:-}"
TIMESTAMP=$(date -u +%Y-%m-%dT%H:%M:%SZ)

if [[ -z "${QLIK_TENANT:-}" ]] || [[ -z "${QLIK_API_KEY:-}" ]]; then
  echo "{\"success\":false,\"error\":\"QLIK_TENANT and QLIK_API_KEY required\",\"timestamp\":\"$TIMESTAMP\"}"
  exit 1
fi

if [[ -z "$ASSISTANT_ID" ]]; then
  echo "{\"success\":false,\"error\":\"Assistant ID required. Usage: qlik-answers-ask.sh <assistant-id> \\\"question\\\" [thread-id]\",\"timestamp\":\"$TIMESTAMP\"}"
  exit 1
fi

if [[ -z "$QUESTION" ]]; then
  echo "{\"success\":false,\"error\":\"Question required. Usage: qlik-answers-ask.sh <assistant-id> \\\"question\\\" [thread-id]\",\"timestamp\":\"$TIMESTAMP\"}"
  exit 1
fi

TENANT="${QLIK_TENANT%/}"
[[ "$TENANT" != http* ]] && TENANT="https://$TENANT"

# If no thread ID, create one first
if [[ -z "$THREAD_ID" ]]; then
  THREAD_NAME="Conversation: ${TIMESTAMP}"
  THREAD_RESPONSE=$(curl -sL -X POST \
    -H "Authorization: Bearer ${QLIK_API_KEY}" \
    -H "Content-Type: application/json" \
    -d "{\"name\": \"${THREAD_NAME}\"}" \
    "${TENANT}/api/v1/assistants/${ASSISTANT_ID}/threads")
  
  THREAD_ID=$(echo "$THREAD_RESPONSE" | python3 -c "import sys,json; d=json.load(sys.stdin); print(d.get('id',''))" 2>/dev/null)
  
  if [[ -z "$THREAD_ID" ]]; then
    echo "{\"success\":false,\"error\":\"Failed to create thread\",\"details\":$THREAD_RESPONSE,\"timestamp\":\"$TIMESTAMP\"}"
    exit 1
  fi
fi

# Invoke the question on the thread
RESPONSE=$(curl -sL -X POST \
  -H "Authorization: Bearer ${QLIK_API_KEY}" \
  -H "Content-Type: application/json" \
  -d "{\"input\":{\"prompt\":$(echo "$QUESTION" | python3 -c 'import sys,json; print(json.dumps(sys.stdin.read().strip()))'),\"promptType\":\"thread\",\"includeText\":true}}" \
  "${TENANT}/api/v1/assistants/${ASSISTANT_ID}/threads/${THREAD_ID}/actions/invoke")

echo "$RESPONSE" | python3 -c "
import json
import sys

question = '''$QUESTION'''
assistant_id = '$ASSISTANT_ID'
thread_id = '$THREAD_ID'
timestamp = '$TIMESTAMP'

try:
    data = json.load(sys.stdin)
    if 'errors' in data:
        print(json.dumps({'success': False, 'error': data['errors'][0].get('title', 'Unknown error'), 'details': data['errors'], 'timestamp': timestamp}, indent=2))
        sys.exit(1)
    
    print(json.dumps({
        'success': True,
        'assistantId': assistant_id,
        'threadId': thread_id,
        'question': question.strip(),
        'answer': data.get('output', data.get('answer', '')),
        'sources': data.get('sources', []),
        'timestamp': timestamp
    }, indent=2))
except Exception as e:
    print(json.dumps({'success': False, 'error': str(e), 'timestamp': timestamp}, indent=2))
    sys.exit(1)
"
