#!/bin/bash
# Qlik Cloud Data Alerts
# List data alerts
# Usage: qlik-alerts.sh [limit]

set -euo pipefail

LIMIT="${1:-50}"
TIMESTAMP=$(date -u +%Y-%m-%dT%H:%M:%SZ)

if [[ -z "${QLIK_TENANT:-}" ]] || [[ -z "${QLIK_API_KEY:-}" ]]; then
  echo "{\"success\":false,\"error\":\"QLIK_TENANT and QLIK_API_KEY required\",\"timestamp\":\"$TIMESTAMP\"}"
  exit 1
fi

TENANT="${QLIK_TENANT%/}"
[[ "$TENANT" != http* ]] && TENANT="https://$TENANT"

curl -sL \
  -H "Authorization: Bearer ${QLIK_API_KEY}" \
  -H "Content-Type: application/json" \
  "${TENANT}/api/v1/data-alerts?limit=${LIMIT}" | python3 -c "
import json
import sys

timestamp = '$TIMESTAMP'

try:
    data = json.load(sys.stdin)
    
    if 'errors' in data:
        print(json.dumps({'success': False, 'error': data['errors'][0].get('title', 'Unknown error'), 'timestamp': timestamp}, indent=2))
        sys.exit(1)
    
    alerts = []
    for a in data.get('data', []):
        alerts.append({
            'id': a.get('id'),
            'name': a.get('name'),
            'description': a.get('description'),
            'enabled': a.get('enabled'),
            'lastTriggerTime': a.get('lastTriggerTime'),
            'ownerId': a.get('ownerId'),
            'appId': a.get('appId')
        })
    
    print(json.dumps({
        'success': True,
        'alerts': alerts,
        'totalCount': len(alerts),
        'timestamp': timestamp
    }, indent=2))
except Exception as e:
    print(json.dumps({'success': False, 'error': str(e), 'timestamp': timestamp}, indent=2))
    sys.exit(1)
"
