#!/bin/bash
# Qlik Cloud User Search
# Search for users by name or email
# Usage: qlik-users-search.sh "query" [limit]

set -euo pipefail

QUERY="${1:-}"
LIMIT="${2:-50}"
TIMESTAMP=$(date -u +%Y-%m-%dT%H:%M:%SZ)

if [[ -z "${QLIK_TENANT:-}" ]] || [[ -z "${QLIK_API_KEY:-}" ]]; then
  echo "{\"success\":false,\"error\":\"QLIK_TENANT and QLIK_API_KEY required\",\"timestamp\":\"$TIMESTAMP\"}"
  exit 1
fi

if [[ -z "$QUERY" ]]; then
  echo "{\"success\":false,\"error\":\"Search query required. Usage: qlik-users-search.sh \\\"query\\\"\",\"timestamp\":\"$TIMESTAMP\"}"
  exit 1
fi

TENANT="${QLIK_TENANT%/}"
[[ "$TENANT" != http* ]] && TENANT="https://$TENANT"

# Build and URL encode the filter using Python
ENCODED_FILTER=$(python3 -c "
import urllib.parse
query = '''$QUERY'''
# Use 'co' (contains) operator for partial matching
filter_str = f'name co \"{query}\"'
print(urllib.parse.quote(filter_str, safe=''))
")

curl -sL \
  -H "Authorization: Bearer ${QLIK_API_KEY}" \
  -H "Content-Type: application/json" \
  "${TENANT}/api/v1/users?filter=${ENCODED_FILTER}&limit=${LIMIT}" | python3 -c "
import json
import sys

query = '''$QUERY'''
timestamp = '$TIMESTAMP'

try:
    raw = sys.stdin.read()
    if not raw.strip():
        print(json.dumps({'success': True, 'query': query, 'users': [], 'totalCount': 0, 'timestamp': timestamp}, indent=2))
        sys.exit(0)
        
    data = json.loads(raw)
    if 'errors' in data:
        print(json.dumps({'success': False, 'error': data['errors'][0].get('title', 'Unknown error'), 'timestamp': timestamp}, indent=2))
        sys.exit(1)
    
    users = []
    for u in data.get('data', []):
        users.append({
            'id': u.get('id'),
            'name': u.get('name'),
            'email': u.get('email'),
            'status': u.get('status')
        })
    
    print(json.dumps({'success': True, 'query': query, 'users': users, 'totalCount': len(users), 'timestamp': timestamp}, indent=2))
except json.JSONDecodeError:
    print(json.dumps({'success': True, 'query': query, 'users': [], 'totalCount': 0, 'timestamp': timestamp}, indent=2))
except Exception as e:
    print(json.dumps({'success': False, 'error': str(e), 'timestamp': timestamp}, indent=2))
    sys.exit(1)
"
