#!/bin/bash
# Qlik Cloud Tenant Info
# Get tenant and current user information
# Usage: qlik-tenant.sh

set -euo pipefail

TIMESTAMP=$(date -u +%Y-%m-%dT%H:%M:%SZ)

if [[ -z "${QLIK_TENANT:-}" ]] || [[ -z "${QLIK_API_KEY:-}" ]]; then
  echo "{\"success\":false,\"error\":\"QLIK_TENANT and QLIK_API_KEY required\",\"timestamp\":\"$TIMESTAMP\"}"
  exit 1
fi

TENANT="${QLIK_TENANT%/}"
[[ "$TENANT" != http* ]] && TENANT="https://$TENANT"

curl -sL \
  -H "Authorization: Bearer ${QLIK_API_KEY}" \
  -H "Content-Type: application/json" \
  "${TENANT}/api/v1/users/me" | python3 -c "
import json
import sys

tenant = '$TENANT'
timestamp = '$TIMESTAMP'

try:
    data = json.load(sys.stdin)
    
    if 'errors' in data:
        print(json.dumps({'success': False, 'error': data['errors'][0].get('title', 'Unknown error'), 'timestamp': timestamp}, indent=2))
        sys.exit(1)
    
    # Extract roles from assignedRoles
    roles = [r.get('name') for r in data.get('assignedRoles', [])]
    
    result = {
        'success': True,
        'platform': 'cloud',
        'tenant': {
            'id': data.get('tenantId'),
            'url': tenant
        },
        'user': {
            'id': data.get('id'),
            'name': data.get('name'),
            'email': data.get('email'),
            'subject': data.get('subject'),
            'status': data.get('status'),
            'roles': roles,
            'created': data.get('createdAt'),
            'lastUpdated': data.get('lastUpdatedAt')
        },
        'timestamp': timestamp
    }
    print(json.dumps(result, indent=2))
except Exception as e:
    print(json.dumps({'success': False, 'error': str(e), 'timestamp': timestamp}, indent=2))
    sys.exit(1)
"
