#!/bin/bash
# Qlik Cloud Reload Status
# Check the status of a reload
# Usage: qlik-reload-status.sh <reload-id>

set -euo pipefail

RELOAD_ID="${1:-}"
TIMESTAMP=$(date -u +%Y-%m-%dT%H:%M:%SZ)

if [[ -z "${QLIK_TENANT:-}" ]] || [[ -z "${QLIK_API_KEY:-}" ]]; then
  echo "{\"success\":false,\"error\":\"QLIK_TENANT and QLIK_API_KEY required\",\"timestamp\":\"$TIMESTAMP\"}"
  exit 1
fi

if [[ -z "$RELOAD_ID" ]]; then
  echo "{\"success\":false,\"error\":\"Reload ID required. Usage: qlik-reload-status.sh <reload-id>\",\"timestamp\":\"$TIMESTAMP\"}"
  exit 1
fi

TENANT="${QLIK_TENANT%/}"
[[ "$TENANT" != http* ]] && TENANT="https://$TENANT"

curl -sL \
  -H "Authorization: Bearer ${QLIK_API_KEY}" \
  -H "Content-Type: application/json" \
  "${TENANT}/api/v1/reloads/${RELOAD_ID}" | python3 -c "
import json
import sys

reload_id = '$RELOAD_ID'
timestamp = '$TIMESTAMP'

try:
    data = json.load(sys.stdin)
    
    if 'errors' in data:
        error = data['errors'][0].get('title', 'Unknown error')
        print(json.dumps({'success': False, 'error': error, 'reloadId': reload_id, 'timestamp': timestamp}, indent=2))
        sys.exit(1)
    
    # Truncate log to avoid huge output
    log = data.get('log', '')
    if len(log) > 500:
        log = log[:500] + '... (truncated)'
    
    result = {
        'success': True,
        'reload': {
            'id': data.get('id'),
            'appId': data.get('appId'),
            'status': data.get('status'),
            'type': data.get('type'),
            'partial': data.get('partial'),
            'creationTime': data.get('creationTime'),
            'startTime': data.get('startTime'),
            'endTime': data.get('endTime'),
            'duration': data.get('duration'),
            'errorCode': data.get('errorCode'),
            'errorMessage': data.get('errorMessage'),
            'log': log
        },
        'timestamp': timestamp
    }
    print(json.dumps(result, indent=2))
except Exception as e:
    print(json.dumps({'success': False, 'error': str(e), 'timestamp': timestamp}, indent=2))
    sys.exit(1)
"
