#!/bin/bash
# Qlik Cloud Reload History
# Get reload history for an app
# Usage: qlik-reload-history.sh <app-id> [limit]

set -euo pipefail

APP_ID="${1:-}"
LIMIT="${2:-10}"
TIMESTAMP=$(date -u +%Y-%m-%dT%H:%M:%SZ)

if [[ -z "${QLIK_TENANT:-}" ]] || [[ -z "${QLIK_API_KEY:-}" ]]; then
  echo "{\"success\":false,\"error\":\"QLIK_TENANT and QLIK_API_KEY required\",\"timestamp\":\"$TIMESTAMP\"}"
  exit 1
fi

if [[ -z "$APP_ID" ]]; then
  echo "{\"success\":false,\"error\":\"App ID required. Usage: qlik-reload-history.sh <app-id> [limit]\",\"timestamp\":\"$TIMESTAMP\"}"
  exit 1
fi

TENANT="${QLIK_TENANT%/}"
[[ "$TENANT" != http* ]] && TENANT="https://$TENANT"

curl -sL \
  -H "Authorization: Bearer ${QLIK_API_KEY}" \
  -H "Content-Type: application/json" \
  "${TENANT}/api/v1/reloads?appId=${APP_ID}&limit=${LIMIT}" | python3 -c "
import json
import sys

app_id = '$APP_ID'
timestamp = '$TIMESTAMP'

try:
    data = json.load(sys.stdin)
    
    if 'errors' in data:
        print(json.dumps({'success': False, 'error': data['errors'][0].get('title', 'Unknown error'), 'timestamp': timestamp}, indent=2))
        sys.exit(1)
    
    reloads = []
    for r in data.get('data', []):
        reloads.append({
            'id': r.get('id'),
            'status': r.get('status'),
            'type': r.get('type'),
            'partial': r.get('partial'),
            'creationTime': r.get('creationTime'),
            'startTime': r.get('startTime'),
            'endTime': r.get('endTime'),
            'duration': r.get('duration'),
            'errorCode': r.get('errorCode')
        })
    
    print(json.dumps({
        'success': True,
        'appId': app_id,
        'reloads': reloads,
        'count': len(reloads),
        'timestamp': timestamp
    }, indent=2))
except Exception as e:
    print(json.dumps({'success': False, 'error': str(e), 'timestamp': timestamp}, indent=2))
    sys.exit(1)
"
