#!/bin/bash
# Qlik Cloud Reload Failures
# Find recent failed reloads across all apps
# Usage: qlik-reload-failures.sh [hours-back]
#
# hours-back: How many hours to look back (default: 24)

set -euo pipefail

HOURS="${1:-24}"
TIMESTAMP=$(date -u +%Y-%m-%dT%H:%M:%SZ)

if [[ -z "${QLIK_TENANT:-}" ]] || [[ -z "${QLIK_API_KEY:-}" ]]; then
  echo "{\"success\":false,\"error\":\"QLIK_TENANT and QLIK_API_KEY required\",\"timestamp\":\"$TIMESTAMP\"}"
  exit 1
fi

TENANT="${QLIK_TENANT%/}"
[[ "$TENANT" != http* ]] && TENANT="https://$TENANT"

curl -sL \
  -H "Authorization: Bearer ${QLIK_API_KEY}" \
  -H "Content-Type: application/json" \
  "${TENANT}/api/v1/reloads?limit=100&status=FAILED" | python3 -c "
import json
import sys
from datetime import datetime, timedelta, timezone

hours = int('$HOURS')
timestamp = '$TIMESTAMP'
cutoff = datetime.now(timezone.utc) - timedelta(hours=hours)

try:
    data = json.load(sys.stdin)
    reloads = data.get('data', [])
    
    failures = []
    for r in reloads:
        created = r.get('creationTime', '')
        if created:
            try:
                dt = datetime.fromisoformat(created.replace('Z', '+00:00'))
                if dt < cutoff:
                    continue
            except:
                pass
        
        failures.append({
            'id': r.get('id'),
            'appId': r.get('appId'),
            'status': r.get('status'),
            'errorCode': r.get('errorCode'),
            'errorMessage': (r.get('errorMessage') or '')[:200],
            'createdAt': r.get('creationTime'),
            'type': r.get('type')
        })
    
    result = {
        'success': True,
        'hoursBack': hours,
        'failureCount': len(failures),
        'failures': failures,
        'timestamp': timestamp
    }
    
    if len(failures) == 0:
        result['message'] = f'No reload failures in the last {hours} hours! 🎉'
    
    print(json.dumps(result, indent=2))
except Exception as e:
    print(json.dumps({'success': False, 'error': str(e), 'timestamp': timestamp}, indent=2))
    sys.exit(1)
"
