#!/bin/bash
# Qlik Cloud Cancel Reload
# Cancel a running reload task
# Usage: qlik-reload-cancel.sh <reload-id>

set -euo pipefail

RELOAD_ID="${1:-}"
TIMESTAMP=$(date -u +%Y-%m-%dT%H:%M:%SZ)

if [[ -z "${QLIK_TENANT:-}" ]] || [[ -z "${QLIK_API_KEY:-}" ]]; then
  echo "{\"success\":false,\"error\":\"QLIK_TENANT and QLIK_API_KEY required\",\"timestamp\":\"$TIMESTAMP\"}"
  exit 1
fi

if [[ -z "$RELOAD_ID" ]]; then
  echo "{\"success\":false,\"error\":\"Reload ID required. Usage: qlik-reload-cancel.sh <reload-id>\",\"timestamp\":\"$TIMESTAMP\"}"
  exit 1
fi

TENANT="${QLIK_TENANT%/}"
[[ "$TENANT" != http* ]] && TENANT="https://$TENANT"

curl -sL -X POST \
  -H "Authorization: Bearer ${QLIK_API_KEY}" \
  -H "Content-Type: application/json" \
  "${TENANT}/api/v1/reloads/${RELOAD_ID}/actions/cancel" | python3 -c "
import json
import sys

reload_id = '$RELOAD_ID'
timestamp = '$TIMESTAMP'

try:
    raw = sys.stdin.read().strip()
    
    # Empty response means success
    if not raw:
        print(json.dumps({'success': True, 'reloadId': reload_id, 'message': 'Reload cancelled', 'timestamp': timestamp}, indent=2))
        sys.exit(0)
    
    data = json.loads(raw)
    
    if 'errors' in data:
        error = data['errors'][0].get('title', 'Unknown error')
        print(json.dumps({'success': False, 'error': error, 'reloadId': reload_id, 'timestamp': timestamp}, indent=2))
        sys.exit(1)
    
    print(json.dumps({'success': True, 'reloadId': reload_id, 'result': data, 'timestamp': timestamp}, indent=2))
except json.JSONDecodeError:
    print(json.dumps({'success': True, 'reloadId': reload_id, 'message': 'Reload cancelled', 'timestamp': timestamp}, indent=2))
except Exception as e:
    print(json.dumps({'success': False, 'error': str(e), 'timestamp': timestamp}, indent=2))
    sys.exit(1)
"
