#!/bin/bash
# Qlik Cloud Data Lineage
# Get lineage for an app or dataset
# Usage: qlik-lineage.sh <app-id|secure-qri|dataset-id> [direction] [levels]
#
# Input types:
#   - App ID (UUID like 950a5da4-0e61-466b-a1c5-805b072da128) → uses app lineage
#   - SecureQri (starts with qri:) → uses lineage-graphs API
#   - Dataset ID → fetches secureQri first, then uses lineage-graphs
#
# direction: upstream, downstream, both (default: both)
# levels: Number of levels, -1 for unlimited (default: 5)

set -euo pipefail

NODE_ID="${1:-}"
DIRECTION="${2:-both}"
LEVELS="${3:-5}"
TIMESTAMP=$(date -u +%Y-%m-%dT%H:%M:%SZ)

if [[ -z "${QLIK_TENANT:-}" ]] || [[ -z "${QLIK_API_KEY:-}" ]]; then
  echo "{\"success\":false,\"error\":\"QLIK_TENANT and QLIK_API_KEY required\",\"timestamp\":\"$TIMESTAMP\"}"
  exit 1
fi

if [[ -z "$NODE_ID" ]]; then
  echo "{\"success\":false,\"error\":\"App ID, Dataset ID, or SecureQri required\",\"timestamp\":\"$TIMESTAMP\"}"
  exit 1
fi

TENANT="${QLIK_TENANT%/}"
[[ "$TENANT" != http* ]] && TENANT="https://$TENANT"

# Check if it's a UUID (app ID format: 8-4-4-4-12)
if [[ "$NODE_ID" =~ ^[0-9a-f]{8}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{12}$ ]]; then
  # It's an app ID - use app lineage endpoint
  curl -sL \
    -H "Authorization: Bearer ${QLIK_API_KEY}" \
    -H "Content-Type: application/json" \
    "${TENANT}/api/v1/apps/${NODE_ID}/data/lineage" | python3 -c "
import json
import sys
import re

app_id = '$NODE_ID'
timestamp = '$TIMESTAMP'

try:
    data = json.load(sys.stdin)
    
    if isinstance(data, dict) and 'errors' in data:
        print(json.dumps({'success': False, 'error': data['errors'][0].get('title', 'Unknown error'), 'appId': app_id, 'timestamp': timestamp}, indent=2))
        sys.exit(1)
    
    sources = []
    internal = []
    
    for item in data if isinstance(data, list) else []:
        disc = item.get('discriminator', '')
        
        if disc.startswith('{lib://'):
            match = re.match(r'\{lib://([^:]+)[^}]*:([^}]+)\}', disc)
            if match:
                connection = match.group(1)
                path = match.group(2)
                filename = path.split('/')[-1]
                ext = filename.split('.')[-1].lower() if '.' in filename else ''
                file_type = 'QVD' if ext == 'qvd' else 'Excel' if ext in ['xlsx', 'xls'] else 'CSV' if ext == 'csv' else 'File'
                sources.append({'type': file_type, 'connection': connection, 'path': path, 'fileName': filename})
        elif disc.startswith('RESIDENT '):
            internal.append({'type': 'Resident', 'table': disc.replace('RESIDENT ', '').replace(';', '')})
        elif disc == 'INLINE;':
            internal.append({'type': 'Inline'})
    
    print(json.dumps({
        'success': True,
        'type': 'app-lineage',
        'appId': app_id,
        'sources': sources,
        'internal': internal,
        'sourceCount': len(sources),
        'internalCount': len(internal),
        'timestamp': timestamp
    }, indent=2))
except Exception as e:
    print(json.dumps({'success': False, 'error': str(e), 'appId': app_id, 'timestamp': timestamp}, indent=2))
    sys.exit(1)
"
  exit $?
fi

# Check if it's a QRI
QRI="$NODE_ID"
if [[ ! "$NODE_ID" =~ ^qri: ]]; then
  # It's a dataset ID - fetch secureQri first
  DATASET=$(curl -sL \
    -H "Authorization: Bearer ${QLIK_API_KEY}" \
    -H "Content-Type: application/json" \
    "${TENANT}/api/v1/data-sets/${NODE_ID}")
  
  QRI=$(echo "$DATASET" | python3 -c "import sys,json; d=json.load(sys.stdin); print(d.get('secureQri', ''))" 2>/dev/null)
  
  if [[ -z "$QRI" ]]; then
    echo "{\"success\":false,\"error\":\"Dataset ${NODE_ID} has no secureQri for lineage\",\"hint\":\"Try using qlik-app-lineage.sh for app lineage instead\",\"timestamp\":\"$TIMESTAMP\"}"
    exit 1
  fi
fi

# URL encode the QRI
ENCODED_QRI=$(python3 -c "import urllib.parse; print(urllib.parse.quote('$QRI', safe=''))")

# Build params based on direction
if [[ "$DIRECTION" == "upstream" ]]; then
  PARAMS="up=${LEVELS}&down=0"
elif [[ "$DIRECTION" == "downstream" ]]; then
  PARAMS="up=0&down=${LEVELS}"
else
  PARAMS="up=${LEVELS}&down=${LEVELS}"
fi
PARAMS="${PARAMS}&level=resource&collapse=false"

curl -sL \
  -H "Authorization: Bearer ${QLIK_API_KEY}" \
  -H "Content-Type: application/json" \
  "${TENANT}/api/v1/lineage-graphs/nodes/${ENCODED_QRI}?${PARAMS}" | python3 -c "
import json
import sys

qri = '''$QRI'''
direction = '$DIRECTION'
levels = '$LEVELS'
timestamp = '$TIMESTAMP'

try:
    data = json.load(sys.stdin)
    
    if 'errors' in data:
        error = data['errors'][0].get('title', data['errors'][0].get('detail', 'Unknown error'))
        print(json.dumps({'success': False, 'error': error, 'qri': qri, 'timestamp': timestamp}, indent=2))
        sys.exit(1)
    
    graph = data.get('graph', {})
    nodes = graph.get('nodes', {})
    edges = graph.get('edges', [])
    
    node_list = []
    for nid, node in (nodes.items() if isinstance(nodes, dict) else []):
        node_list.append({
            'id': nid,
            'resourceType': node.get('resourceType'),
            'name': node.get('name'),
            'subType': node.get('subType')
        })
    
    result = {
        'success': True,
        'type': 'dataset-lineage',
        'qri': qri,
        'direction': direction,
        'levels': int(levels),
        'graph': {
            'nodeCount': len(nodes) if isinstance(nodes, dict) else len(nodes),
            'edgeCount': len(edges),
            'nodes': node_list[:20],
            'edges': edges[:30]
        },
        'timestamp': timestamp
    }
    
    print(json.dumps(result, indent=2))
except Exception as e:
    print(json.dumps({'success': False, 'error': str(e), 'qri': qri, 'timestamp': timestamp}, indent=2))
    sys.exit(1)
"
