#!/bin/bash
# Qlik Cloud Insight Advisor (Natural Language Query)
# Ask questions about your data in natural language - returns actual data!
# Usage: qlik-insight.sh "question" [app-id]

set -euo pipefail

QUESTION="${1:-}"
APP_ID="${2:-}"
TIMESTAMP=$(date -u +%Y-%m-%dT%H:%M:%SZ)

if [[ -z "${QLIK_TENANT:-}" ]] || [[ -z "${QLIK_API_KEY:-}" ]]; then
  echo "{\"success\":false,\"error\":\"QLIK_TENANT and QLIK_API_KEY required\",\"timestamp\":\"$TIMESTAMP\"}"
  exit 1
fi

if [[ -z "$QUESTION" ]]; then
  echo "{\"success\":false,\"error\":\"Question required. Usage: qlik-insight.sh \\\"question\\\" [app-id]\",\"timestamp\":\"$TIMESTAMP\"}"
  exit 1
fi

TENANT="${QLIK_TENANT%/}"
[[ "$TENANT" != http* ]] && TENANT="https://$TENANT"

# Build request body with visualization options for actual data
REQUEST_BODY=$(printf '%s' "$QUESTION" | python3 -c "
import json
import sys
question = sys.stdin.read()
body = {
    'text': question,
    'enableVisualizations': True,
    'visualizationOptions': {'includeCellData': True}
}
app_id = '$APP_ID'
if app_id:
    body['app'] = {'id': app_id}
print(json.dumps(body))
")

curl -sL -X POST \
  -H "Authorization: Bearer ${QLIK_API_KEY}" \
  -H "Content-Type: application/json" \
  -d "$REQUEST_BODY" \
  "${TENANT}/api/v1/questions/actions/ask" | QUESTION="$QUESTION" APP_ID="$APP_ID" python3 -c "
import json
import sys
import os

question = os.environ.get('QUESTION', '')
app_id = os.environ.get('APP_ID', '')
timestamp = '$TIMESTAMP'

try:
    data = json.load(sys.stdin)
    if 'errors' in data:
        print(json.dumps({'success': False, 'error': data['errors'][0].get('title', 'Unknown error'), 'timestamp': timestamp}, indent=2))
        sys.exit(1)
    
    result = {
        'success': True,
        'question': question,
        'timestamp': timestamp
    }
    
    if 'conversationalResponse' in data:
        resp = data['conversationalResponse']
        
        # Get narrative (actual insights with numbers)
        for r in resp.get('responses', []):
            if 'narrative' in r:
                result['narrative'] = r['narrative'].get('text', '')
            if 'chartData' in r:
                result['chartData'] = r['chartData']
        
        # Get app info
        if resp.get('apps'):
            result['app'] = {
                'id': resp['apps'][0].get('id'),
                'name': resp['apps'][0].get('name')
            }
        
        # Get recommendations
        if resp.get('recommendations'):
            result['recommendations'] = [
                {'name': rec.get('name'), 'id': rec.get('recId')}
                for rec in resp['recommendations'][:5]
            ]
        
        # Get drill-down link
        if resp.get('drillDownURI'):
            result['drillDownLink'] = resp['drillDownURI']
    
    print(json.dumps(result, indent=2))
except Exception as e:
    print(json.dumps({'success': False, 'error': str(e), 'timestamp': timestamp}, indent=2))
    sys.exit(1)
"
