#!/bin/bash
# Qlik Cloud Health Check
# Check connectivity and get basic tenant/user info
# Usage: qlik-health.sh

set -euo pipefail

TIMESTAMP=$(date -u +%Y-%m-%dT%H:%M:%SZ)

if [[ -z "${QLIK_TENANT:-}" ]] || [[ -z "${QLIK_API_KEY:-}" ]]; then
  echo "{\"success\":false,\"error\":\"QLIK_TENANT and QLIK_API_KEY required\",\"timestamp\":\"$TIMESTAMP\"}"
  exit 1
fi

TENANT="${QLIK_TENANT%/}"
[[ "$TENANT" != http* ]] && TENANT="https://$TENANT"

curl -sL \
  -H "Authorization: Bearer ${QLIK_API_KEY}" \
  -H "Content-Type: application/json" \
  "${TENANT}/api/v1/users/me" | python3 -c "
import json
import sys

tenant = '$TENANT'
timestamp = '$TIMESTAMP'

try:
    data = json.load(sys.stdin)
    
    if 'errors' in data:
        print(json.dumps({'success': False, 'error': data['errors'][0].get('title', 'Unknown error'), 'timestamp': timestamp}, indent=2))
        sys.exit(1)
    
    result = {
        'success': True,
        'status': 'healthy',
        'platform': 'cloud',
        'tenant': {
            'id': data.get('tenantId'),
            'url': tenant
        },
        'user': {
            'id': data.get('id'),
            'name': data.get('name'),
            'email': data.get('email')
        },
        'services': {
            'api': 'operational'
        },
        'timestamp': timestamp
    }
    print(json.dumps(result, indent=2))
except Exception as e:
    print(json.dumps({'success': False, 'error': str(e), 'status': 'unhealthy', 'timestamp': timestamp}, indent=2))
    sys.exit(1)
"
