#!/bin/bash
# Qlik Cloud Dataset Details (Managed Data)
# Get detailed information about a managed dataset
# Usage: qlik-dataset-get.sh <dataset-id>

set -euo pipefail

DATASET_ID="${1:-}"
TIMESTAMP=$(date -u +%Y-%m-%dT%H:%M:%SZ)

if [[ -z "${QLIK_TENANT:-}" ]] || [[ -z "${QLIK_API_KEY:-}" ]]; then
  echo "{\"success\":false,\"error\":\"QLIK_TENANT and QLIK_API_KEY required\",\"timestamp\":\"$TIMESTAMP\"}"
  exit 1
fi

if [[ -z "$DATASET_ID" ]]; then
  echo "{\"success\":false,\"error\":\"Dataset ID required. Usage: qlik-dataset-get.sh <dataset-id>\",\"timestamp\":\"$TIMESTAMP\"}"
  exit 1
fi

TENANT="${QLIK_TENANT%/}"
[[ "$TENANT" != http* ]] && TENANT="https://$TENANT"

curl -sL \
  -H "Authorization: Bearer ${QLIK_API_KEY}" \
  -H "Content-Type: application/json" \
  "${TENANT}/api/v1/data-sets/${DATASET_ID}" | python3 -c "
import json
import sys

dataset_id = '$DATASET_ID'
timestamp = '$TIMESTAMP'

try:
    data = json.load(sys.stdin)
    
    if 'errors' in data:
        print(json.dumps({'success': False, 'error': data['errors'][0].get('title', 'Unknown error'), 'datasetId': dataset_id, 'timestamp': timestamp}, indent=2))
        sys.exit(1)
    
    op = data.get('operational', {})
    tech = data.get('technicalMetadata', data.get('technical', {}))
    schema = data.get('schema', {})
    
    # Extract field info
    fields = []
    for f in schema.get('dataFields', []):
        fields.append({
            'name': f.get('name'),
            'type': f.get('dataType') or f.get('type'),
            'nullable': f.get('nullable'),
            'primaryKey': f.get('primaryKey')
        })
    
    result = {
        'success': True,
        'dataset': {
            'id': data.get('id'),
            'name': data.get('name'),
            'description': data.get('description'),
            'type': data.get('type', op.get('logicalType')),
            'spaceId': data.get('spaceId'),
            'ownerId': data.get('ownerId'),
            'secureQri': data.get('secureQri'),
            'size': op.get('size') or op.get('sizeBytes') or tech.get('size'),
            'rowCount': op.get('rowCount') or op.get('recordCount') or tech.get('rowCount'),
            'columnCount': op.get('columnCount') or op.get('fieldCount') or tech.get('columnCount'),
            'fields': fields[:20],  # First 20 fields
            'fieldCount': len(fields),
            'createdAt': data.get('createdAt'),
            'updatedAt': data.get('updatedAt')
        },
        'timestamp': timestamp
    }
    if len(fields) > 20:
        result['note'] = f'Showing first 20 of {len(fields)} fields'
    
    print(json.dumps(result, indent=2))
except Exception as e:
    print(json.dumps({'success': False, 'error': str(e), 'timestamp': timestamp}, indent=2))
    sys.exit(1)
"
