#!/bin/bash
# Qlik Cloud Data File Details
# Get detailed information about a data file
# Usage: qlik-datafile.sh <data-file-id>

set -euo pipefail

FILE_ID="${1:-}"
TIMESTAMP=$(date -u +%Y-%m-%dT%H:%M:%SZ)

if [[ -z "${QLIK_TENANT:-}" ]] || [[ -z "${QLIK_API_KEY:-}" ]]; then
  echo "{\"success\":false,\"error\":\"QLIK_TENANT and QLIK_API_KEY required\",\"timestamp\":\"$TIMESTAMP\"}"
  exit 1
fi

if [[ -z "$FILE_ID" ]]; then
  echo "{\"success\":false,\"error\":\"Data file ID required. Usage: qlik-dataset.sh <data-file-id>\",\"timestamp\":\"$TIMESTAMP\"}"
  exit 1
fi

TENANT="${QLIK_TENANT%/}"
[[ "$TENANT" != http* ]] && TENANT="https://$TENANT"

curl -sL \
  -H "Authorization: Bearer ${QLIK_API_KEY}" \
  -H "Content-Type: application/json" \
  "${TENANT}/api/v1/data-files/${FILE_ID}" | python3 -c "
import json
import sys

timestamp = '$TIMESTAMP'

try:
    data = json.load(sys.stdin)
    
    if 'errors' in data:
        print(json.dumps({'success': False, 'error': data['errors'][0].get('title', 'Unknown error'), 'timestamp': timestamp}, indent=2))
        sys.exit(1)
    
    result = {
        'success': True,
        'dataFile': {
            'id': data.get('id'),
            'name': data.get('name'),
            'baseName': data.get('baseName'),
            'folder': data.get('folder', False),
            'size': data.get('size'),
            'spaceId': data.get('spaceId'),
            'ownerId': data.get('ownerId'),
            'qri': data.get('qri'),
            'actions': data.get('actions', []),
            'createdDate': data.get('createdDate'),
            'modifiedDate': data.get('modifiedDate'),
            'contentUpdatedDate': data.get('contentUpdatedDate')
        },
        'timestamp': timestamp
    }
    print(json.dumps(result, indent=2))
except Exception as e:
    print(json.dumps({'success': False, 'error': str(e), 'timestamp': timestamp}, indent=2))
    sys.exit(1)
"
