#!/bin/bash
# Qlik Cloud AutoML Experiments
# List AutoML experiments (Cloud-only)
# Usage: qlik-automl-experiments.sh [limit]

set -euo pipefail

LIMIT="${1:-50}"
TIMESTAMP=$(date -u +%Y-%m-%dT%H:%M:%SZ)

if [[ -z "${QLIK_TENANT:-}" ]] || [[ -z "${QLIK_API_KEY:-}" ]]; then
  echo "{\"success\":false,\"error\":\"QLIK_TENANT and QLIK_API_KEY required\",\"timestamp\":\"$TIMESTAMP\"}"
  exit 1
fi

TENANT="${QLIK_TENANT%/}"
[[ "$TENANT" != http* ]] && TENANT="https://$TENANT"

curl -sL \
  -H "Authorization: Bearer ${QLIK_API_KEY}" \
  -H "Content-Type: application/json" \
  "${TENANT}/api/v1/ml/experiments?limit=${LIMIT}" | python3 -c "
import json
import sys

timestamp = '$TIMESTAMP'

try:
    data = json.load(sys.stdin)
    if 'errors' in data:
        print(json.dumps({'success': False, 'error': data['errors'][0].get('title', 'Unknown error'), 'timestamp': timestamp}, indent=2))
        sys.exit(1)
    
    experiments = []
    for e in data.get('data', []):
        attrs = e.get('attributes', {})
        experiments.append({
            'id': e.get('id'),
            'name': attrs.get('name'),
            'description': attrs.get('description'),
            'status': attrs.get('status'),
            'targetColumn': attrs.get('targetColumn'),
            'spaceId': attrs.get('spaceId'),
            'ownerId': attrs.get('ownerId'),
            'created': attrs.get('createdAt'),
            'updated': attrs.get('updatedAt')
        })
    
    total = data.get('meta', {}).get('count', len(experiments))
    print(json.dumps({'success': True, 'experiments': experiments, 'totalCount': total, 'timestamp': timestamp}, indent=2))
except Exception as e:
    print(json.dumps({'success': False, 'error': str(e), 'timestamp': timestamp}, indent=2))
    sys.exit(1)
"
