#!/bin/bash
# Qlik Cloud AutoML Experiment
# Get details of a specific experiment
# Usage: qlik-automl-experiment.sh <experiment-id>

set -euo pipefail

EXPERIMENT_ID="${1:-}"
TIMESTAMP=$(date -u +%Y-%m-%dT%H:%M:%SZ)

if [[ -z "${QLIK_TENANT:-}" ]] || [[ -z "${QLIK_API_KEY:-}" ]]; then
  echo "{\"success\":false,\"error\":\"QLIK_TENANT and QLIK_API_KEY required\",\"timestamp\":\"$TIMESTAMP\"}"
  exit 1
fi

if [[ -z "$EXPERIMENT_ID" ]]; then
  echo "{\"success\":false,\"error\":\"Experiment ID required. Usage: qlik-automl-experiment.sh <experiment-id>\",\"timestamp\":\"$TIMESTAMP\"}"
  exit 1
fi

TENANT="${QLIK_TENANT%/}"
[[ "$TENANT" != http* ]] && TENANT="https://$TENANT"

curl -sL \
  -H "Authorization: Bearer ${QLIK_API_KEY}" \
  -H "Content-Type: application/json" \
  "${TENANT}/api/v1/ml/experiments/${EXPERIMENT_ID}" | python3 -c "
import json
import sys

timestamp = '$TIMESTAMP'

try:
    data = json.load(sys.stdin)
    
    if 'errors' in data:
        print(json.dumps({'success': False, 'error': data['errors'][0].get('title', 'Unknown error'), 'timestamp': timestamp}, indent=2))
        sys.exit(1)
    
    # Handle JSON:API format
    if 'data' in data:
        attrs = data['data'].get('attributes', {})
        exp_id = data['data'].get('id')
    else:
        attrs = data
        exp_id = data.get('id')
    
    result = {
        'success': True,
        'experiment': {
            'id': exp_id,
            'name': attrs.get('name'),
            'description': attrs.get('description'),
            'status': attrs.get('status'),
            'targetColumn': attrs.get('targetColumn'),
            'spaceId': attrs.get('spaceId'),
            'ownerId': attrs.get('ownerId'),
            'created': attrs.get('createdAt'),
            'updated': attrs.get('updatedAt')
        },
        'timestamp': timestamp
    }
    print(json.dumps(result, indent=2))
except Exception as e:
    print(json.dumps({'success': False, 'error': str(e), 'timestamp': timestamp}, indent=2))
    sys.exit(1)
"
