#!/bin/bash
# Qlik Cloud AutoML Deployments
# List ML model deployments (Cloud-only)
# Usage: qlik-automl-deployments.sh [limit]

set -euo pipefail

LIMIT="${1:-50}"
TIMESTAMP=$(date -u +%Y-%m-%dT%H:%M:%SZ)

if [[ -z "${QLIK_TENANT:-}" ]] || [[ -z "${QLIK_API_KEY:-}" ]]; then
  echo "{\"success\":false,\"error\":\"QLIK_TENANT and QLIK_API_KEY required\",\"timestamp\":\"$TIMESTAMP\"}"
  exit 1
fi

TENANT="${QLIK_TENANT%/}"
[[ "$TENANT" != http* ]] && TENANT="https://$TENANT"

curl -sL \
  -H "Authorization: Bearer ${QLIK_API_KEY}" \
  -H "Content-Type: application/json" \
  "${TENANT}/api/v1/ml/deployments?limit=${LIMIT}" | python3 -c "
import json
import sys

timestamp = '$TIMESTAMP'

try:
    data = json.load(sys.stdin)
    if 'errors' in data:
        print(json.dumps({'success': False, 'error': data['errors'][0].get('title', 'Unknown error'), 'timestamp': timestamp}, indent=2))
        sys.exit(1)
    
    deployments = []
    for d in data.get('data', []):
        attrs = d.get('attributes', {})
        deployments.append({
            'id': d.get('id'),
            'name': attrs.get('name'),
            'description': attrs.get('description'),
            'modelId': attrs.get('modelId'),
            'experimentId': attrs.get('experimentId'),
            'spaceId': attrs.get('spaceId'),
            'ownerId': attrs.get('ownerId'),
            'enablePredictions': attrs.get('enablePredictions'),
            'deprecated': attrs.get('deprecated'),
            'created': attrs.get('createdAt'),
            'updated': attrs.get('updatedAt')
        })
    
    total = data.get('meta', {}).get('count', len(deployments))
    print(json.dumps({'success': True, 'deployments': deployments, 'totalCount': total, 'timestamp': timestamp}, indent=2))
except Exception as e:
    print(json.dumps({'success': False, 'error': str(e), 'timestamp': timestamp}, indent=2))
    sys.exit(1)
"
