#!/bin/bash
# Qlik Cloud Run Automation
# Execute an automation
# Usage: qlik-automation-run.sh <automation-id>

set -euo pipefail

AUTOMATION_ID="${1:-}"
TIMESTAMP=$(date -u +%Y-%m-%dT%H:%M:%SZ)

if [[ -z "${QLIK_TENANT:-}" ]] || [[ -z "${QLIK_API_KEY:-}" ]]; then
  echo "{\"success\":false,\"error\":\"QLIK_TENANT and QLIK_API_KEY required\",\"timestamp\":\"$TIMESTAMP\"}"
  exit 1
fi

if [[ -z "$AUTOMATION_ID" ]]; then
  echo "{\"success\":false,\"error\":\"Automation ID required. Usage: qlik-automation-run.sh <automation-id>\",\"timestamp\":\"$TIMESTAMP\"}"
  exit 1
fi

TENANT="${QLIK_TENANT%/}"
[[ "$TENANT" != http* ]] && TENANT="https://$TENANT"

curl -sL -X POST \
  -H "Authorization: Bearer ${QLIK_API_KEY}" \
  -H "Content-Type: application/json" \
  "${TENANT}/api/v1/automations/${AUTOMATION_ID}/actions/run" | python3 -c "
import json
import sys

automation_id = '$AUTOMATION_ID'
timestamp = '$TIMESTAMP'

try:
    raw = sys.stdin.read().strip()
    
    if not raw:
        print(json.dumps({'success': True, 'automationId': automation_id, 'message': 'Automation triggered', 'timestamp': timestamp}, indent=2))
        sys.exit(0)
    
    data = json.loads(raw)
    
    if 'errors' in data:
        print(json.dumps({'success': False, 'error': data['errors'][0].get('title', 'Unknown error'), 'timestamp': timestamp}, indent=2))
        sys.exit(1)
    
    result = {
        'success': True,
        'automationId': automation_id,
        'run': {
            'id': data.get('id'),
            'status': data.get('status'),
            'startTime': data.get('startTime')
        },
        'message': 'Automation run started',
        'timestamp': timestamp
    }
    print(json.dumps(result, indent=2))
except json.JSONDecodeError:
    print(json.dumps({'success': True, 'automationId': automation_id, 'message': 'Automation triggered', 'timestamp': timestamp}, indent=2))
except Exception as e:
    print(json.dumps({'success': False, 'error': str(e), 'timestamp': timestamp}, indent=2))
    sys.exit(1)
"
