#!/bin/bash
# Qlik Cloud Apps List
# List all apps accessible to the current user
# Usage: qlik-apps.sh [limit]

set -euo pipefail

LIMIT="${1:-50}"
TIMESTAMP=$(date -u +%Y-%m-%dT%H:%M:%SZ)

if [[ -z "${QLIK_TENANT:-}" ]] || [[ -z "${QLIK_API_KEY:-}" ]]; then
  echo "{\"success\":false,\"error\":\"QLIK_TENANT and QLIK_API_KEY required\",\"timestamp\":\"$TIMESTAMP\"}"
  exit 1
fi

TENANT="${QLIK_TENANT%/}"
[[ "$TENANT" != http* ]] && TENANT="https://$TENANT"

# Get apps list and pipe directly to Python
curl -sL \
  -H "Authorization: Bearer ${QLIK_API_KEY}" \
  -H "Content-Type: application/json" \
  "${TENANT}/api/v1/apps?limit=${LIMIT}" | python3 -c "
import json
import sys

timestamp = '$TIMESTAMP'

try:
    data = json.load(sys.stdin)
    if 'errors' in data:
        print(json.dumps({'success': False, 'error': data['errors'][0].get('title', 'Unknown error'), 'timestamp': timestamp}, indent=2))
        sys.exit(1)
    
    apps = []
    for item in data.get('data', []):
        attrs = item.get('attributes', item)
        apps.append({
            'id': attrs.get('id'),
            'name': attrs.get('name'),
            'description': attrs.get('description', '')[:100] if attrs.get('description') else None,
            'spaceId': attrs.get('spaceId'),
            'published': attrs.get('published'),
            'lastReloadTime': attrs.get('lastReloadTime'),
            'modified': attrs.get('modifiedDate')
        })
    
    print(json.dumps({'success': True, 'apps': apps, 'totalCount': len(apps), 'timestamp': timestamp}, indent=2))
except Exception as e:
    print(json.dumps({'success': False, 'error': str(e), 'timestamp': timestamp}, indent=2))
    sys.exit(1)
"
