#!/bin/bash
# Qlik Cloud App Data Lineage
# Get data sources/connections for an app
# Usage: qlik-app-lineage.sh <app-id>

set -euo pipefail

APP_ID="${1:-}"
TIMESTAMP=$(date -u +%Y-%m-%dT%H:%M:%SZ)

if [[ -z "${QLIK_TENANT:-}" ]] || [[ -z "${QLIK_API_KEY:-}" ]]; then
  echo "{\"success\":false,\"error\":\"QLIK_TENANT and QLIK_API_KEY required\",\"timestamp\":\"$TIMESTAMP\"}"
  exit 1
fi

if [[ -z "$APP_ID" ]]; then
  echo "{\"success\":false,\"error\":\"App ID required. Usage: qlik-app-lineage.sh <app-id>\",\"timestamp\":\"$TIMESTAMP\"}"
  exit 1
fi

TENANT="${QLIK_TENANT%/}"
[[ "$TENANT" != http* ]] && TENANT="https://$TENANT"

curl -sL \
  -H "Authorization: Bearer ${QLIK_API_KEY}" \
  -H "Content-Type: application/json" \
  "${TENANT}/api/v1/apps/${APP_ID}/data/lineage" | python3 -c "
import json
import sys
import re

app_id = '$APP_ID'
timestamp = '$TIMESTAMP'

try:
    data = json.load(sys.stdin)
    
    if isinstance(data, dict) and 'errors' in data:
        print(json.dumps({'success': False, 'error': data['errors'][0].get('title', 'Unknown error'), 'appId': app_id, 'timestamp': timestamp}, indent=2))
        sys.exit(1)
    
    sources = []
    internal = []
    
    for item in data if isinstance(data, list) else []:
        disc = item.get('discriminator', '')
        
        # External data files (lib://)
        if disc.startswith('{lib://'):
            match = re.match(r'\{lib://([^:]+)[^}]*:([^}]+)\}', disc)
            if match:
                connection = match.group(1)
                path = match.group(2)
                filename = path.split('/')[-1]
                ext = filename.split('.')[-1].lower() if '.' in filename else ''
                
                file_type = 'QVD' if ext == 'qvd' else 'Excel' if ext in ['xlsx', 'xls'] else 'CSV' if ext == 'csv' else 'File'
                sources.append({
                    'type': file_type,
                    'connection': connection,
                    'path': path,
                    'fileName': filename,
                    'extension': ext
                })
        
        # Database connections
        elif 'database' in disc.lower() or disc.startswith('ODBC'):
            sources.append({'type': 'Database', 'discriminator': disc[:100]})
        
        # Resident tables
        elif disc.startswith('RESIDENT '):
            internal.append({'type': 'Resident', 'table': disc.replace('RESIDENT ', '').replace(';', '')})
        
        # Inline data
        elif disc == 'INLINE;':
            internal.append({'type': 'Inline'})
    
    print(json.dumps({
        'success': True,
        'appId': app_id,
        'sources': sources,
        'internal': internal,
        'sourceCount': len(sources),
        'internalCount': len(internal),
        'timestamp': timestamp
    }, indent=2))
except Exception as e:
    print(json.dumps({'success': False, 'error': str(e), 'appId': app_id, 'timestamp': timestamp}, indent=2))
    sys.exit(1)
"
